package com.digiwin.dap.middleware.dict.api;

import com.digiwin.dap.middleware.dict.domain.DictDTO;
import com.digiwin.dap.middleware.dict.entity.Dict;
import com.digiwin.dap.middleware.dict.service.DictService;
import com.digiwin.dap.middleware.domain.StdData;
import com.github.pagehelper.PageSerializable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Optional;

/**
 * 字典表控制层
 *
 * @author chenzhuang
 * @since 2022-09-05 13:47:36
 */
@RestController
@RequestMapping("/api/${spring.application.name}/v2/dict")
public class DictController{
    @Autowired
    private DictService dictService;

    /**
     * 新增字典类型
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/add")
    public StdData<Integer> saveDict(@RequestBody DictDTO dto) {
        Assert.hasText(dto.getId(), "id不能为空");
        Assert.hasText(dto.getName(), "名称不能为空");
        return StdData.ofSuccess(dictService.insertDict(dto.doForward()));
    }

    /**
     * 修改字典类型
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/mod")
    public StdData<Integer> updateDict(@RequestBody DictDTO dto) {
        Assert.notNull(dto.getSid(), "sid不能为空");
        Assert.hasText(dto.getId(), "id不能为空");
        Assert.hasText(dto.getName(), "名称不能为空");
        return StdData.ofSuccess(dictService.updateDict(dto.doForward()));
    }

    /**
     * 删除字典类型（关联删除对应的字典数据）
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/del")
    public StdData<Integer> deleteDict(@RequestBody DictDTO dto) {
        Assert.notNull(dto.getSid(), "字典sid不能为空");
        return StdData.ofSuccess(dictService.deleteDictBySid(dto.getSid()));
    }

    /**
     * 查询字典类型
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/find")
    public StdData<Dict> findDict(@RequestBody DictDTO dto) {
        Assert.isTrue(dto.getSid() != null || StringUtils.hasLength(dto.getId()), "字典sid、id不能全为空");
        return StdData.ofSuccess(dictService.selectDict(dto));
    }

    /**
     * 校验字典类型id是否存在
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/exists")
    public StdData<Boolean> existsDict(@RequestBody DictDTO dto) {
        Assert.hasText(dto.getId(), "id不能为空");
        return StdData.ofSuccess(Optional.ofNullable(dictService.selectDictById(dto.getId())).isPresent());
    }

    /**
     * 查询字典类型[分页]
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/search")
    public StdData<PageSerializable<Dict>> searchDict(@RequestBody DictDTO dto) {
        return StdData.ofSuccess(dictService.selectDictList(dto, dto.getPageNum(), dto.getPageSize(), dto.getOrderBy()));
    }

    /**
     * 查询字典类型列表
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/list")
    public StdData<List<Dict>> listDict(@RequestBody DictDTO dto) {
        return StdData.ofSuccess(dictService.selectDictList(dto));
    }
}

