package com.digiwin.dap.middleware.dict.api;

import com.digiwin.dap.middleware.dict.domain.DictDataDTO;
import com.digiwin.dap.middleware.dict.entity.DictData;
import com.digiwin.dap.middleware.dict.service.DictDataService;
import com.digiwin.dap.middleware.domain.StdData;
import com.github.pagehelper.PageSerializable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Optional;

/**
 * 字典数据表控制层
 *
 * @author chenzhuang
 * @since 2022-09-05 13:47:36
 */
@RestController
@RequestMapping("/api/${spring.application.name}/v2/dict/data")
public class DictDataController {
    @Autowired
    private DictDataService dictDataService;

    /**
     * 新增字典数据
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/add")
    public StdData<Integer> saveDictData(@RequestBody DictDataDTO dto) {
        Assert.hasText(dto.getDictId(), "字典id不能为空");
        Assert.hasText(dto.getDictKey(), "字典key不能为空");
        Assert.hasText(dto.getDictValue(), "字典值不能为空");
        return StdData.ofSuccess(dictDataService.insertDictData(dto.doForward()));
    }

    /**
     * 修改字典数据
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/mod")
    public StdData<Integer> updateDictData(@RequestBody DictDataDTO dto) {
        Assert.notNull(dto.getSid(), "sid不能为空");
        Assert.hasText(dto.getDictId(), "字典id不能为空");
        Assert.hasText(dto.getDictKey(), "字典项文本不能为空");
        Assert.hasText(dto.getDictValue(), "字典项值不能为空");
        return StdData.ofSuccess(dictDataService.updateDictData(dto.doForward()));
    }

    /**
     * 删除字典数据
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/del")
    public StdData<Integer> deleteDictData(@RequestBody DictDataDTO dto) {
        Assert.notNull(dto.getSid(), "字典数据sid不能为空");
        return StdData.ofSuccess(dictDataService.deleteDictDataBySid(dto.getSid()));
    }

    /**
     * 查询字典数据
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/find")
    public StdData<DictData> findDictData(@RequestBody DictDataDTO dto) {
        Assert.isTrue(dto.getSid() != null || (StringUtils.hasLength(dto.getDictId()) && StringUtils.hasLength(dto.getDictKey())), "字典sid不能为空");
        Assert.isTrue(dto.getSid() != null || StringUtils.hasLength(dto.getDictId()), "字典id不能为空");
        Assert.isTrue(dto.getSid() != null || StringUtils.hasLength(dto.getDictKey()), "字典key不能为空");
        return StdData.ofSuccess(dictDataService.selectDictData(dto));
    }

    /**
     * 校验字典数据是否存在
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/exists")
    public StdData<Boolean> existsDict(@RequestBody DictDataDTO dto) {
        Assert.hasText(dto.getDictId(), "字典id不能为空");
        Assert.hasText(dto.getDictKey(), "字典key不能为空");
        return StdData.ofSuccess(Optional.ofNullable(dictDataService.selectDictData(dto)).isPresent());
    }

    /**
     * 查询字典数据[分页]
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/search")
    public StdData<PageSerializable<DictData>> searchDictData(@RequestBody DictDataDTO dto) {
        return StdData.ofSuccess(dictDataService.selectDictDataList(dto, dto.getPageNum(), dto.getPageSize(), Optional.ofNullable(dto.getOrderBy()).orElse("dd.dict_sort")));
    }

    /**
     * 查询字典数据列表
     *
     * @param dto the dto
     * @return StdData<Dict>
     */
    @PostMapping("/list")
    public StdData<List<DictData>> listDictData(@RequestBody DictDataDTO dto) {
        return StdData.ofSuccess(dictDataService.selectDictDataList(dto));
    }
}

