package com.digiwin.dap.middle.encrypt.support;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.digiwin.dap.middle.encrypt.constant.EncryptConstants;
import com.digiwin.dap.middle.encrypt.domain.AppSecretVO;
import com.digiwin.dap.middleware.cache.RedisUtils;
import com.digiwin.dap.middleware.constant.GlobalConstants;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.domain.StdData;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.util.JsonUtils;
import com.digiwin.dap.middleware.util.StringUtils;
import com.digiwin.dap.middleware.util.UserUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * 默认获取appSecret
 *
 * @author ChenZhuang
 * @date @date 2024-1-22 17:14:07
 */
public class DefaultDapSecretSupport implements DapSecretSupport {

    protected static final String APP_SECRET_CURRENT = "/api/iam/v2/dev/app/secret/current";
    protected static final String ENABLE_APP_SECRET_CURRENT = "/api/iam/v2/dev/app/enable/secret/current";
    protected static final String TENANT_SECRET_CURRENT = "/api/iam/v1/isv/credential/secret/current";

    private final DapEnv dapEnv;
    private final RestTemplate restTemplate;

    public DefaultDapSecretSupport(DapEnv dapEnv, RestTemplate restTemplate) {
        this.dapEnv = dapEnv;
        this.restTemplate = restTemplate;
    }

    /**
     * 获取应用appSecret
     * <p>
     * TODO 获取当前应用appSecret，白名单调用有一定风险
     *
     * @param userToken 用户token
     * @param appToken  应用token
     * @return 返回应用appSecret
     */
    @Override
    public String getAppSecret(String userToken, String appToken) {
        DecodedJWT jwt = JWT.decode(appToken);
        String id = jwt.getClaim("id").asString();
        String appSecretKey = String.format(EncryptConstants.REDIS_APP_SECRET, id);
        String appSecret = RedisUtils.get(appSecretKey, String.class);
        if (ObjectUtils.isEmpty(appSecret)) {
            String url = dapEnv.getIamUri() + APP_SECRET_CURRENT;
            HttpHeaders headers = new HttpHeaders();
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
            headers.add(GlobalConstants.HTTP_HEADER_APP_TOKEN_KEY, appToken);
            HttpEntity<Object> request = new HttpEntity<>(headers);
            try {
                ResponseEntity<StdData<AppSecretVO>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, request, new ParameterizedTypeReference<StdData<AppSecretVO>>() {
                });
                StdData<AppSecretVO> stdData = responseEntity.getBody();
                if (stdData == null || !stdData.getSuccess()) {
                    return null;
                }
                return Optional.ofNullable(stdData.getData()).map(AppSecretVO::getAppSecret).orElse(null);
            } catch (Exception e) {
                // 弹窗就使用默认BUSINESS【作业处理异常，请联系客服专员】
                throw new BusinessException(String.format("【调用IAM】获取应用[%s]appSecret失败", UserUtils.getSysId()), e);
            }
        }
        return appSecret;
    }

    @Override
    public List<String> getEnableAppSecret(String userToken, String appToken) {
        DecodedJWT jwt = JWT.decode(appToken);
        String id = jwt.getClaim("id").asString();
        String appSecretKey = String.format(EncryptConstants.REDIS_ENABLE_APP_SECRET_LIST, id);
        String appSecretListStr = RedisUtils.get(appSecretKey, String.class);
        List<Map<String,String>> appSecretMapList = new ArrayList<>();
        if(StringUtils.hasLength(appSecretListStr)){
            appSecretMapList = JsonUtils.jsonToObj(appSecretListStr,  new TypeReference<List<Map<String,String>>>() {});
        }

        if (CollectionUtils.isEmpty(appSecretMapList)) {
            String url = dapEnv.getIamUri() + ENABLE_APP_SECRET_CURRENT;
            HttpHeaders headers = new HttpHeaders();
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
            headers.add(GlobalConstants.HTTP_HEADER_APP_TOKEN_KEY, appToken);
            HttpEntity<Object> request = new HttpEntity<>(headers);
            try {
                ResponseEntity<StdData<AppSecretVO>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, request, new ParameterizedTypeReference<StdData<AppSecretVO>>() {
                });
                StdData<AppSecretVO> stdData = responseEntity.getBody();
                if (stdData == null || !stdData.getSuccess()) {
                    return null;
                }
                appSecretMapList = Optional.ofNullable(stdData.getData()).map(AppSecretVO::getAppSecretList).orElse(null);
            } catch (Exception e) {
                // 弹窗就使用默认BUSINESS【作业处理异常，请联系客服专员】
                throw new BusinessException(String.format("【调用IAM】获取应用[%s]appSecret失败", UserUtils.getSysId()), e);
            }
        }
        return filterExpireAppSecret(appSecretMapList);
    }

    @Override
    public String getDevSecret(String userToken, String devKey) {
        String url = dapEnv.getIamUri() + TENANT_SECRET_CURRENT;
        try {
            Map<String, Object> body = new HashMap<>();
            body.put("devKey", devKey);
            HttpHeaders headers = new HttpHeaders();
            headers.add(GlobalConstants.HTTP_HEADER_USER_TOKEN_KEY, userToken);
            HttpEntity<Object> request = new HttpEntity<>(body, headers);

            ResponseEntity<StdData<AppSecretVO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, request, new ParameterizedTypeReference<StdData<AppSecretVO>>() {
            });
            return responseEntity.getBody().getData().getAppSecret();
        } catch (Exception e) {
            // 弹窗就使用默认BUSINESS【作业处理异常，请联系客服专员】
            throw new BusinessException(String.format("【调用IAM】获取租户[%s]devSecret失败", UserUtils.getTenantId()), e);
        }
    }

    private List<String> filterExpireAppSecret(List<Map<String, String>> appSecretMapList){
        if(CollectionUtils.isEmpty(appSecretMapList)){
            return new ArrayList<>();
        }
        List<String> result = new ArrayList<>();
        for (Map<String, String> appSecretMap : appSecretMapList) {
            String expireTime = appSecretMap.get("expireTime");
            if(!StringUtils.hasLength(expireTime) || LocalDateTime.now().isBefore(LocalDateTime.parse(expireTime, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")))){
                //如果过期时间为空，则说明是永久有效
                result.add(appSecretMap.get("appSecret"));
            }
        }
        return result;
    }
}
