package com.digiwin.dap.middle.encrypt.config;

import com.digiwin.dap.middle.encrypt.AnnotationUtil;
import com.digiwin.dap.middle.encrypt.constant.EncryptConstants;
import com.digiwin.dap.middle.encrypt.domain.annotation.DapEncrypt;
import com.digiwin.dap.middle.encrypt.domain.annotation.DapSign;
import com.digiwin.dap.middleware.commons.crypto.AES;
import com.digiwin.dap.middleware.commons.crypto.SignUtils;
import com.digiwin.dap.middleware.domain.StdData;
import com.digiwin.dap.middleware.domain.StdDataEncrypt;
import com.digiwin.dap.middleware.util.JsonUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.core.MethodParameter;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.http.server.ServletServerHttpRequest;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import javax.servlet.http.HttpServletRequest;

/**
 * StdData 加密加签 响应处理
 * <p>
 * 根据验签、解密的标记，自动加密、加签
 *
 * @author ChenZhuang
 * @date 2024 -3-7 14:12:18
 */
@ControllerAdvice
public class StdDataEncryptResponse implements ResponseBodyAdvice<StdData> {
    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        boolean hasAnnotation = AnnotationUtil.getAnnotation(returnType, DapEncrypt.class) != null
                || AnnotationUtil.getAnnotation(returnType, DapSign.class) != null;
        // 检查返回类型是否为 ResponseEntity 或其子类
        boolean isResponseTypeSupported = StdData.class.isAssignableFrom(returnType.getParameterType());
        return hasAnnotation && isResponseTypeSupported;
    }

    @Override
    public StdData<?> beforeBodyWrite(StdData body, MethodParameter returnType, MediaType selectedContentType, Class<? extends HttpMessageConverter<?>> selectedConverterType, ServerHttpRequest request, ServerHttpResponse response) {
        if (body == null || body.getData() == null) {
            return body;
        }
        HttpServletRequest req = ((ServletServerHttpRequest) request).getServletRequest();
        DapEncrypt dapEncrypt = AnnotationUtil.getAnnotation(returnType, DapEncrypt.class);
        if (dapEncrypt != null && dapEncrypt.response()) {
            String appSecret = (String) req.getAttribute(EncryptConstants.APP_SECRET_KEY);
            String dataString;
            if (BeanUtils.isSimpleProperty(body.getData().getClass())) {
                dataString = body.getData().toString();
            } else {
                dataString = JsonUtils.objToJson(body.getData());
            }
            String eData = AES.encryptIvCBC(dataString, appSecret);
            body.setData(eData);
        }

        DapSign dapSign = AnnotationUtil.getAnnotation(returnType, DapSign.class);
        if (dapSign != null && dapSign.response()) {
            StdDataEncrypt<Object> dataEncrypt = new StdDataEncrypt<>();
            dataEncrypt.setCode(body.getCode());
            dataEncrypt.setMessage(body.getMessage());
            dataEncrypt.setSuccess(body.getSuccess());
            dataEncrypt.setData(body.getData());

            String appSecret = (String) req.getAttribute(EncryptConstants.APP_SECRET_KEY);
            String dataString;
            if (BeanUtils.isSimpleProperty(body.getData().getClass())) {
                dataString = body.getData().toString();
            } else {
                dataString = SignUtils.sortParam(body.getData());
            }
            String sign = SignUtils.sign(appSecret, dataString);
            dataEncrypt.setSign(sign);
            body = dataEncrypt;
        }
        return body;
    }
}