package com.digiwin.dap.middle.gateway.service.track;

import com.digiwin.dap.middle.gateway.constant.GatewayConstants;
import com.digiwin.dap.middle.gateway.service.track.http.HttpTrack;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.lmc.LMC;
import com.digiwin.dap.middleware.lmc.request.SaveEventLog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

@Service
public class LmcHttpTrackService {

    private static final Logger logger = LoggerFactory.getLogger(LmcHttpTrackService.class);

    @Autowired
    private DapEnv dapEnv;
    @Autowired
    private LMC gwLmcClient;

    public void save(List<HttpTrack> tracks) {
        if (tracks == null || tracks.isEmpty()) {
            return;
        }
        List<SaveEventLog> eventLogs = new ArrayList<>();
        for (HttpTrack track : tracks) {
            SaveEventLog eventLog = new SaveEventLog();
            eventLog.setAppId(GatewayConstants.HTTP_TRACK_CR_STATS_CENTER);
            eventLog.setEventSource(dapEnv.getAppName());
            eventLog.setStatus(HttpStatus.valueOf(track.getResponse().getStatus()).is2xxSuccessful() ? 0 : 1);
            eventLog.setEventType(11);
            eventLog.setEventId("SERVICE_CALL");
            eventLog.setEventName("服务调用");
            if (track.getUser() != null) {
                eventLog.setUserId(track.getUser().getUserId());
                eventLog.setUserName(track.getUser().getUserName());
                eventLog.setTenantId(track.getUser().getTenantId());
                eventLog.setTenantName(track.getUser().getTenantName());
            }
            if (track.getSys() != null) {
                eventLog.setSysId(track.getSys().getId());
            }

            // 租户、应用默认值处理
            if (StrUtils.isEmpty(eventLog.getTenantId())) {
                eventLog.setTenantId(GatewayConstants.UNKNOWN);
            }
            if (StrUtils.isEmpty(eventLog.getSysId())) {
                eventLog.setSysId(GatewayConstants.UNKNOWN);
            }

            // 请求信息处理
            HttpTrack.Request request = track.getRequest();
            if (request != null) {
                eventLog.setMethod(request.getMethod());
                eventLog.setRequestUrl(request.getUri().getPath());

                Map<String, Object> contents = new LinkedHashMap<>();
                ZonedDateTime invokeTime = track.getTimestamp().atZone(ZoneOffset.systemDefault());
                contents.put("invokeTime", invokeTime.format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS")));
                contents.put("method", request.getMethod());
                contents.put("uri", request.getUri());
                contents.put("headers", new HashMap<>(request.getHeaders()));
                contents.put("remoteAddress", request.getRemoteAddress());
                contents.put("timeTaken", track.getTimeTaken());
                eventLog.setContent(contents);
            }
            eventLogs.add(eventLog);
        }
        try {
            gwLmcClient.saveEventLogBatch(eventLogs);
            logger.debug("Http访问记录入库：成功处理[{}]条数据", tracks.size());
        } catch (Exception e) {
            logger.error("记录访问明细异常", e);
        }
    }
}
