package com.digiwin.dap.middle.ha.service;

import com.digiwin.dap.middle.ha.config.HealthRedisTemplate;
import com.digiwin.dap.middle.ha.domain.HA;
import com.digiwin.dap.middle.ha.domain.Redis;
import com.digiwin.dap.middleware.domain.DapEnv;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionCommands;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * Redis切换服务
 *
 * @author fobgochod
 * @date 2021/2/2
 */
@Service
public class RedisHealthService implements ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(RedisHealthService.class);
    private static final HA ha = new HA();

    @Autowired
    private DapEnv dapEnv;
    @Autowired(required = false)
    private RedisMailService redisMailService;
    @Autowired(required = false)
    private RedisConfigService redisConfigService;
    @Autowired(required = false)
    private HealthRedisTemplate healthRedisTemplate;

    @Scheduled(fixedRate = 5000)
    public void execute() {
        logger.debug("Redis故障监控中...");
        try {
            healthRedisTemplate.execute(RedisConnectionCommands::ping);
            this.toMaster();
        } catch (Exception e) {
            this.toReplica();
        }
    }

    /**
     * 切换回主节点
     */
    public void toMaster() {
        if (!redisConfigService.isMasterRedis()) {
            logger.info("Redis第{}次访问正常，连续{}次正常判定为可用，将自动切换回Master节点", ++HA.OK_COUNT, HA.MAX_OK_COUNT);
            if (HA.OK_COUNT == 1) {
                ha.setNormalDate(LocalDateTime.now());
            }
            if (HA.OK_COUNT >= HA.MAX_OK_COUNT) {
                if (redisConfigService.changeHost(redisConfigService.master())) {
                    // 切换成功， 发送邮件
                    HA.OK_COUNT = 0;
                    logger.warn("Redis恢复正常，切换回Master节点：{}", redisConfigService.master());
                    ha.setSwitchDate(LocalDateTime.now());
                    ha.setTip("Redis恢复正常，切换回Master节点");
                    // sendMail
                    this.sendMail();
                } else {
                    logger.error("Redis第{}次访问正常，切换回Master节点失败", HA.OK_COUNT);
                }
            }
        }
        HA.FAIL_COUNT = 0;
    }

    private void sendMail() {
        if (redisMailService != null) {
            redisMailService.switchRedis(RedisHealthService.ha);
        }
    }

    /**
     * 切换到备用节点
     */
    private void toReplica() {
        if (redisConfigService.isMasterRedis()) {
            logger.info("Redis第{}次访问异常，连续{}次异常判定为故障，将自动切换到Replica节点", ++HA.FAIL_COUNT, HA.MAX_FAIL_COUNT);
            if (HA.FAIL_COUNT == 1) {
                Redis master = redisConfigService.master();
                Redis replica = redisConfigService.replica();

                ha.setHost(dapEnv.getHost());
                ha.setIp(dapEnv.getIpAddress());
                ha.setFailDate(LocalDateTime.now());
                ha.setMasterUrl(String.format("%s:%s", master.getHost(), master.getPort()));
                ha.setReplicaUrl(String.format("%s:%s", replica.getHost(), replica.getPort()));
            }
            if (HA.FAIL_COUNT >= HA.MAX_FAIL_COUNT) {
                if (!this.replicaOk()) {
                    logger.error("Redis的Replica节点异常，暂不切换");
                    return;
                }
                if (redisConfigService.changeHost(redisConfigService.replica())) {
                    // 切换成功， 发送邮件
                    HA.FAIL_COUNT = 0;
                    logger.warn("Redis发生故障，切换到Replica节点：{}", redisConfigService.replica());
                    ha.setNormalDate(null);
                    ha.setSwitchDate(LocalDateTime.now());
                    ha.setTip("Redis发生故障，切换到Replica节点");
                    // sendMail
                    this.sendMail();
                } else {
                    logger.error("Redis第{}次访问异常，切换到Replica节点失败", HA.FAIL_COUNT);
                }
            }
        }
        HA.OK_COUNT = 0;
    }

    /**
     * 切换到Replica节点前，判断是否可用
     *
     * @return Boolean
     */
    private boolean replicaOk() {
        JedisConnectionFactory redisFactory = new JedisConnectionFactory(redisConfigService.getReplicaConfig());
        RedisConnection connection = null;
        try {
            connection = RedisConnectionUtils.getConnection(redisFactory);
            return Boolean.TRUE;
        } catch (Exception e) {
            logger.error("Redis的Replica节点异常：{}", e.getMessage());
        } finally {
            RedisConnectionUtils.releaseConnection(connection, redisFactory);
        }
        return Boolean.FALSE;
    }

    @Override
    public void setApplicationContext(ApplicationContext context) throws BeansException {
        try {
            this.redisMailService = context.getBean(RedisMailService.class);
        } catch (NoSuchBeanDefinitionException ignored) {
        }
    }
}
