package com.digiwin.dap.middleware.language.service.impl;

import com.digiwin.dap.middleware.language.domain.LanguageConstants;
import com.digiwin.dap.middleware.language.entity.LanguageDefault;
import com.digiwin.dap.middleware.language.entity.LanguageResource;
import com.digiwin.dap.middleware.language.mapper.LanguageMapper;
import com.digiwin.dap.middleware.language.repository.LanguageDefaultRepository;
import com.digiwin.dap.middleware.language.service.LanguageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
public class LanguageServiceImpl implements LanguageService {

    private static final Logger logger = LoggerFactory.getLogger(LanguageServiceImpl.class);
    private static final String PRIMARY_FIELD_NAME = "sid";
    private static final String ID_FIELD_NAME = "id";
    private static List<LanguageDefault> languageDefaultList;
    @Autowired
    private LanguageDefaultRepository languageDefaultRepository;
    @Autowired
    private LanguageMapper languageMapper;

    private static List<Field> getFields(Collection<String> filedNames, Class<?> valueType) {
        List<Field> fields = new ArrayList<>();
        for (String fieldName : filedNames) {
            Field field = ReflectionUtils.findField(valueType, fieldName);
            if (field != null) {
                fields.add(field);
            }
        }
        return fields;
    }

    private static <T> String getIdValue(T target, Class<T> targetType) {
        return getIdValue(target, ID_FIELD_NAME, targetType);
    }

    private static <T> String getIdValue(T target, String idName, Class<T> targetType) {
        if (!StringUtils.hasLength(idName)) {
            return null;
        }
        Field idField = ReflectionUtils.findField(targetType, idName);
        if (idField == null) {
            logger.error("指定的id字段[{}]不存在 target={}", idName, target);
            return null;
        }
        ReflectionUtils.makeAccessible(idField);
        return (String) ReflectionUtils.getField(idField, target);
    }

    private static <T> Long getSid(T target, Class<T> targetType) {
        return getSid(target, PRIMARY_FIELD_NAME, targetType);
    }

    private static <T> Long getSid(T target, String primaryName, Class<T> targetType) {
        Field primaryField = ReflectionUtils.findField(targetType, primaryName);
        if (primaryField == null) {
            throw new RuntimeException(String.format("指定的主键字段[%s]不存在", primaryName));
        }
        ReflectionUtils.makeAccessible(primaryField);
        Long sid = (Long) ReflectionUtils.getField(primaryField, target);
        return Optional.ofNullable(sid).orElse(LanguageConstants.ZERO);
    }

    @PostConstruct
    public void init() {
        languageDefaultList = languageDefaultRepository.findAll();
    }

    @Override
    public <T> T parse(T target, Class<T> targetType) {
        return this.parse(target, getSid(target, targetType), targetType);
    }

    @Override
    public <T, K> T parse(T target, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, getSid(target, targetType), targetType, entityType);
    }

    @Override
    public <T> T parse(T target, long dataSid, Class<T> targetType) {
        return this.parse(target, dataSid, new HashMap<>(), LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> T parse(T target, long dataSid, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, dataSid, getIdValue(target, targetType), new HashMap<>(), LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> T parse(T target, String primaryName, Class<T> targetType) {
        return this.parse(target, primaryName, new HashMap<>(), LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> T parse(T target, String primaryName, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, primaryName, ID_FIELD_NAME, new HashMap<>(), LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> T parse(T target, long dataSid, String languageTag, Class<T> targetType) {
        return this.parse(target, dataSid, new HashMap<>(), languageTag, targetType);
    }

    @Override
    public <T, K> T parse(T target, long dataSid, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, dataSid, getIdValue(target, targetType), new HashMap<>(), languageTag, targetType, entityType);
    }

    @Override
    public <T> T parse(T target, String primaryName, String languageTag, Class<T> targetType) {
        return this.parse(target, primaryName, new HashMap<>(), languageTag, targetType);
    }

    @Override
    public <T, K> T parse(T target, String primaryName, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, primaryName, ID_FIELD_NAME, new HashMap<>(), languageTag, targetType, entityType);
    }

    @Override
    public <T> T parse(T target, String primaryName, Map<String, String> fieldMap, Class<T> targetType) {
        return this.parse(target, primaryName, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> T parse(T target, String primaryName, Map<String, String> fieldMap, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, primaryName, ID_FIELD_NAME, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> T parse(T target, long dataSid, Map<String, String> fieldMap, Class<T> targetType) {
        return this.parse(target, dataSid, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> T parse(T target, long dataSid, Map<String, String> fieldMap, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, dataSid, getIdValue(target, ID_FIELD_NAME, targetType), fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> T parse(T target, long dataSid, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(target, dataSid, null, fieldMap, languageTag, targetType, null);
    }

    @Override
    public <T> T parse(T target, String primaryName, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(target, primaryName, null, fieldMap, languageTag, targetType, null);
    }

    @Override
    public <T, K> T parse(T target, String primaryName, String idName, Map<String, String> fieldMap, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(target, getSid(target, primaryName, targetType), getIdValue(target, idName, targetType), fieldMap, languageTag, targetType, entityType);
    }

    @Override
    public <T, K> T parse(T target, long dataSid, String idValue, Map<String, String> fieldMap, String languageTag, Class<T> targetType, Class<K> entityType) {
        List<LanguageResource> languages = languageMapper.findByDataSidAndLanguage(dataSid, languageTag);
        // 获取多语言字段名
        if (fieldMap.isEmpty()) {
            languages.stream().map(LanguageResource::getFieldName).distinct().forEach(p -> fieldMap.put(p, p));
        }
        // fieldName->content 属性对应多语言
        Map<String, String> valueMap = languages.stream().collect(Collectors.toMap(LanguageResource::getFieldName, LanguageResource::getContent, (a, b) -> b));
        // 填充默认多语言资料
        Map<String, String> baseValueMap = getBaseValueMap(fieldMap, languageTag, entityType);
        // 获取多语言字段
        List<Field> fields = getFields(fieldMap.keySet(), targetType);
        // 填充字段多语言
        for (Field field : fields) {
            try {
                ReflectionUtils.makeAccessible(field);
                String value = valueMap.get(fieldMap.get(field.getName()));
                if (StringUtils.hasLength(value)) {
                    field.set(target, value);
                } else if (entityType != null && idValue != null) {
                    value = baseValueMap.get(entityType.getSimpleName() + "." + idValue + "-" + fieldMap.get(field.getName()));
                    if (StringUtils.hasLength(value)) {
                        field.set(target, value);
                    }
                }
            } catch (Exception e) {
                logger.error("多语言字段：[{}->{}]，失败原因：{}", field.getName(), fieldMap.get(field.getName()), e.getMessage());
            }
        }
        return target;
    }

    @Override
    public <T> List<T> parse(List<T> targets, Class<T> targetType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, targetType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, new HashMap<>(), targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, Map<String, String> fieldMap, Class<T> targetType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, Map<String, String> fieldMap, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, ID_FIELD_NAME, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, String primaryName, Class<T> targetType) {
        return this.parse(targets, primaryName, LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, String primaryName, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, primaryName, LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, Locale locale, Class<T> targetType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, new HashMap<>(), locale.toLanguageTag(), targetType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, Locale locale, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, ID_FIELD_NAME, new HashMap<>(), locale.toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, String primaryName, Map<String, String> fieldMap, Class<T> targetType) {
        return this.parse(targets, primaryName, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, String primaryName, Map<String, String> fieldMap, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, primaryName, ID_FIELD_NAME, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, String primaryName, String languageTag, Class<T> targetType) {
        return this.parse(targets, primaryName, new HashMap<>(), languageTag, targetType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, String primaryName, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, primaryName, ID_FIELD_NAME, new HashMap<>(), languageTag, targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, List<Long> dataSids, Map<String, String> fieldMap, Class<T> targetType) {
        return parse(targets, dataSids, fieldMap, LocaleContextHolder.getLocale().toLanguageTag(), targetType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, List<Long> dataSids, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, dataSids, fieldMap, languageTag, targetType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, null, new ArrayList<>(), fieldMap, languageTag, targetType, null);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, Map<String, String> fieldMap, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, PRIMARY_FIELD_NAME, ID_FIELD_NAME, new ArrayList<>(), fieldMap, languageTag, targetType, entityType);
    }

    @Override
    public <T> List<T> parse(List<T> targets, String primaryName, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(targets, primaryName, null, new ArrayList<>(), fieldMap, languageTag, targetType, null);
    }

    @Override
    public <T> List<T> parse(List<T> targets, String primaryName, List<Long> dataSids, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(targets, primaryName, null, dataSids, fieldMap, languageTag, targetType, null);
    }

    @Override
    public <T> List<T> parse(List<T> targets, Field primaryField, List<Long> dataSids, Map<String, String> fieldMap, String languageTag, Class<T> targetType) {
        return this.parse(targets, primaryField, null, dataSids, fieldMap, languageTag, targetType, null);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, String primaryName, String idName, Map<String, String> fieldMap, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, getPrimaryField(targets, primaryName, targetType), getIdField(targets, idName, targetType), new ArrayList<>(), fieldMap, languageTag, targetType, entityType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, String primaryName, String idName, List<Long> dataSids, Map<String, String> fieldMap, String languageTag, Class<T> targetType, Class<K> entityType) {
        return this.parse(targets, getPrimaryField(targets, primaryName, targetType), getIdField(targets, idName, targetType), dataSids, fieldMap, languageTag, targetType, entityType);
    }

    @Override
    public <T, K> List<T> parse(List<T> targets, Field primaryField, Field idField, List<Long> dataSids, Map<String, String> fieldMap, String languageTag, Class<T> targetType, Class<K> entityType) {
        if (CollectionUtils.isEmpty(targets) || primaryField == null) {
            return targets;
        }
        ReflectionUtils.makeAccessible(primaryField);
        // 获取主键集合
        if (CollectionUtils.isEmpty(dataSids)) {
            dataSids = targets.stream().flatMap(x -> Stream.of((Long) ReflectionUtils.getField(primaryField, x))).collect(Collectors.toList());
        }
        // 查询目标集合多语言设定
        List<LanguageResource> languages = languageMapper.findByDataSidsAndLanguage(dataSids, languageTag);
        // 获取多语言字段名
        if (fieldMap.isEmpty()) {
            languages.stream().map(LanguageResource::getFieldName).distinct().forEach(p -> fieldMap.put(p, p));
        }
        // 获取数据对应多语言信息
        // bugfix: 微软正式空指针异常
        Map<String, String> valueMap = languages.stream().filter((a) -> a.getContent() != null).
                collect(Collectors.toMap(p -> p.getDataSid() + "-" + p.getFieldName(), LanguageResource::getContent, (a, b) -> b));
        // 填充默认多语言资料
        Map<String, String> baseValueMap = getBaseValueMap(fieldMap, languageTag, entityType);
        // 获取多语言字段
        List<Field> fields = getFields(fieldMap.keySet(), targetType);
        Optional.ofNullable(idField).ifPresent(ReflectionUtils::makeAccessible);
        // 填充字段多语言
        for (T target : targets) {
            Object primaryValue = ReflectionUtils.getField(primaryField, target);
            Object idValue = idField == null ? null : ReflectionUtils.getField(idField, target);
            for (Field field : fields) {
                try {
                    ReflectionUtils.makeAccessible(field);
                    String value = valueMap.get(primaryValue + "-" + fieldMap.get(field.getName()));
                    if (StringUtils.hasLength(value)) {
                        field.set(target, value);
                    } else if (entityType != null && idValue != null) {
                        // 基础多语言
                        value = baseValueMap.get(entityType.getSimpleName() + "." + idValue + "-" + fieldMap.get(field.getName()));
                        if (StringUtils.hasLength(value)) {
                            field.set(target, value);
                        }
                    }
                } catch (Exception e) {
                    logger.error("主键信息：[{}]({})，多语言字段：[{}->{}]，失败原因：{}", primaryField.getName(), primaryValue, field.getName(), fieldMap.get(field.getName()), e.getMessage());
                }
            }
        }
        return targets;
    }

    private <K> Map<String, String> getBaseValueMap(Map<String, String> fieldMap, String languageTag, Class<K> entityType) {
        if (entityType == null) {
            return Collections.emptyMap();
        }
        List<LanguageDefault> baseList = languageDefaultList.stream().filter(x -> Objects.equals(languageTag, x.getLanguage()) && x.getDataId().startsWith(entityType.getSimpleName() + ".")).collect(Collectors.toList());
        baseList.stream().map(LanguageDefault::getFieldName).distinct().forEach(baseField -> {
            if (!fieldMap.containsValue(baseField)) {
                fieldMap.put(baseField, baseField);
            }
        });
        return baseList.stream().collect(Collectors.toMap(o -> o.getDataId() + "-" + o.getFieldName(), LanguageDefault::getContent, (a, b) -> b));
    }

    private <T> Field getPrimaryField(List<T> targets, Class<T> targetType) {
        Field primaryField = ReflectionUtils.findField(targetType, PRIMARY_FIELD_NAME);
        if (primaryField == null) {
            logger.error("指定的主键字段[{}]不存在 target={}", PRIMARY_FIELD_NAME, CollectionUtils.isEmpty(targets) ? null : targets.get(0));
        }
        return primaryField;
    }

    private <T> Field getPrimaryField(List<T> targets, String primaryName, Class<T> targetType) {
        Field primaryField = ReflectionUtils.findField(targetType, primaryName);
        if (primaryField == null) {
            logger.error("指定的主键字段[{}]不存在 target={}", primaryName, CollectionUtils.isEmpty(targets) ? null : targets.get(0));
        }
        return primaryField;
    }

    private <T> Field getIdField(List<T> targets, Class<T> targetType) {
        Field idField = ReflectionUtils.findField(targetType, ID_FIELD_NAME);
        if (idField == null) {
            logger.error("指定的id字段[{}]不存在 target={}, ", ID_FIELD_NAME, CollectionUtils.isEmpty(targets) ? null : targets.get(0));
        }
        return idField;
    }

    private <T> Field getIdField(List<T> targets, String idName, Class<T> targetType) {
        if (!StringUtils.hasLength(idName)) {
            return null;
        }
        Field idField = ReflectionUtils.findField(targetType, idName);
        if (idField == null) {
            logger.error("指定的id字段[{}]不存在 target={}, ", idName, CollectionUtils.isEmpty(targets) ? null : targets.get(0));
        }
        return idField;
    }
}
