package com.digiwin.dap.middleware.mojo;

import com.spotify.docker.client.AnsiProgressHandler;
import com.spotify.docker.client.DefaultDockerClient;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.exceptions.DockerCertificateException;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.DirectoryScanner;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

/**
 * 构建中间件镜像文件
 *
 * @author fobgo
 * @date 2022/2/16 9:18
 */
@Mojo(name = "build")
public class BuildMojo extends AbstractDapMojo {

    @Parameter(defaultValue = "${project.build.directory}")
    private String buildDirectory;
    @Parameter(defaultValue = "${project.build.finalName}.jar")
    private String jarName;
    @Parameter(property = "dockerDirectory", defaultValue = "${project.basedir}/src/main/resources/docker")
    private String dockerDirectory;

    @Override
    public void execute() throws MojoExecutionException {
        try {
            final Path targetPath = getTargetPath();
            copyFiles(targetPath);
            DockerClient docker = buildClient();
            buildImage(docker, targetPath);
        } catch (Exception e) {
            getLog().error("Build image fail.", e);
        }
    }

    private void copyFiles(Path targetPath) throws IOException {
        // ${project.basedir}/src/main/resources/docker  => ${project.build.directory}/docker
        final DirectoryScanner scanner = new DirectoryScanner();
        scanner.setBasedir(new File(dockerDirectory));
        scanner.scan();
        for (String included : scanner.getIncludedFiles()) {
            final Path source = Paths.get(dockerDirectory).resolve(included);
            final Path target = targetPath.resolve(included);
            getLog().info(String.format("Copying %s -> %s", source, target));
            // ensure all directories exist because copy operation will fail if they don't
            Files.createDirectories(target.getParent());
            Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        }

        // ${project.build.directory}/boot-example-1.0.0.jar => ${project.build.directory}/docker/boot-example-1.0.0.jar
        final Path source = Paths.get(buildDirectory).resolve(jarName);
        final Path target = targetPath.resolve(jarName);
        getLog().info(String.format("Copying %s -> %s", source, target));
        Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
    }

    private DockerClient buildClient() throws MojoExecutionException {
        final DefaultDockerClient.Builder builder;
        try {
            builder = DefaultDockerClient.fromEnv().readTimeoutMillis(0);
            builder.uri(String.format("http://%s:2375", host));
        } catch (DockerCertificateException ex) {
            throw new MojoExecutionException("Cannot build DockerClient due to certificate problem", ex);
        }
        return builder.build();
    }

    private void buildImage(final DockerClient docker, final Path buildDir) throws Exception {
        String fullImageName = imageName + ":" + imageTag;
        getLog().info("Building image " + fullImageName);
        docker.build(buildDir, fullImageName, new AnsiProgressHandler());
        getLog().info("Built " + fullImageName);
    }

    private Path getTargetPath() {
        return Paths.get(buildDirectory, "docker");
    }
}
