package com.digiwin.dap.middleware.mojo;

import com.digiwin.dap.middleware.ssh.DockerUtils;
import com.digiwin.dap.middleware.ssh.Shell;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.DirectoryScanner;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;

/**
 * 一步到位
 * 编译、打包、镜像
 *
 * @author fobgo
 * @date 2022/2/17 8:52
 */
//@Mojo(name = "one-stop")
public class OneStopMojo extends AbstractDapMojo {

    @Parameter(defaultValue = "${project.build.directory}")
    private String buildDirectory;
    @Parameter(defaultValue = "${project.basedir}/")
    private String baseDir;
    @Parameter(defaultValue = "${project.basedir}/src/")
    private String srcDir;

    /**
     * @param sourcePath  被压缩打包的文件夹
     * @param tarFileName 压缩之后的输出文件名称
     */
    public static void compressionTar(String sourcePath, String tarFileName) throws Exception {
        Path source = Paths.get(sourcePath);
        //如果不是文件夹抛出异常
        if (!Files.isDirectory(source)) {
            throw new IOException("请指定一个文件夹");
        }

        //OutputStream输出流、BufferedOutputStream缓冲输出流
        //GzipCompressorOutputStream是gzip压缩输出流
        //TarArchiveOutputStream打tar包输出流（包含gzip压缩输出流）
        try (OutputStream fOut = Files.newOutputStream(Paths.get(tarFileName)); BufferedOutputStream buffOut = new BufferedOutputStream(fOut); GzipCompressorOutputStream gzOut = new GzipCompressorOutputStream(buffOut); TarArchiveOutputStream tOut = new TarArchiveOutputStream(gzOut)) {
            //遍历文件目录树
            Files.walkFileTree(source, new SimpleFileVisitor<Path>() {

                //当成功访问到一个文件
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attributes) throws IOException {

                    // 判断当前遍历文件是不是符号链接(快捷方式)，不做打包压缩处理
                    if (attributes.isSymbolicLink()) {
                        return FileVisitResult.CONTINUE;
                    }

                    //获取当前遍历文件名称
                    Path targetFile = source.relativize(file);

                    //将该文件打包压缩
                    TarArchiveEntry tarEntry = new TarArchiveEntry(file.toFile(), targetFile.toString());
                    tOut.putArchiveEntry(tarEntry);
                    Files.copy(file, tOut);
                    tOut.closeArchiveEntry();
                    //继续下一个遍历文件处理
                    return FileVisitResult.CONTINUE;
                }

                //当前遍历文件访问失败
                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    System.err.printf("无法对该文件压缩打包为tar.gz : %s%n%s%n", file, exc);
                    return FileVisitResult.CONTINUE;
                }

            });
            //for循环完成之后，finish-tar包输出流
            tOut.finish();
        }
    }

    @Override
    public void execute() throws MojoExecutionException {
        try {
            // final Path targetPath = getTargetPath();
            // copyFiles(targetPath);

            String fileName = buildDirectory + "/" + imageName + "-" + imageTag + ".tar.gz";
            compressionTar(srcDir, fileName);
            DockerUtils.instance(new Shell(host, port, username, password)).build(baseDir, imageName, imageTag);
        } catch (Exception e) {
            getLog().error("Build image fail.", e);
        }
    }

    private void copyFiles(Path targetPath) throws IOException {
        // ${project.basedir}/src  => ${project.build.directory}/docker/src
        final DirectoryScanner scanner = new DirectoryScanner();
        scanner.setBasedir(new File(srcDir));
        scanner.scan();
        for (String included : scanner.getIncludedFiles()) {
            final Path source = Paths.get(srcDir).resolve(included);
            final Path target = Paths.get(targetPath.toString(), "src").resolve(included);
            getLog().info(String.format("Copying %s -> %s", source, target));
            // ensure all directories exist because copy operation will fail if they don't
            Files.createDirectories(target.getParent());
            Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        }
        // ${project.basedir}/Dockerfile => ${project.build.directory}/docker/Dockerfile
        String dockerfile = "Dockerfile";
        copFile(targetPath, dockerfile);
        // ${project.basedir}/pom.xml => ${project.build.directory}/docker/pom.xml
        String pomXml = "pom.xml";
        copFile(targetPath, pomXml);
    }

    private void copFile(Path targetPath, String fileName) throws IOException {
        final Path source = Paths.get(baseDir).resolve(fileName);
        final Path target = targetPath.resolve(fileName);
        getLog().info(String.format("Copying %s -> %s", source, target));
        Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
    }

    private Path getTargetPath() {
        return Paths.get(buildDirectory, "docker");
    }
}
