package com.digiwin.dap.middleware.mojo;

import com.digiwin.dap.middleware.domain.Middle;
import com.digiwin.dap.middleware.ssh.DockerUtils;
import com.digiwin.dap.middleware.ssh.Shell;
import org.apache.commons.cli.*;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.List;
import java.util.Optional;

/**
 * 启动容器
 *
 * @author fobgochod
 * @date 2022/2/16 9:17
 */
@Mojo(name = "deploy")
public class DeployMojo extends AbstractDapMojo {

    @Parameter(property = "startFile", defaultValue = "doc/run/startup.sh")
    private String startFile;

    private static String toDuration(long time) {
        long second = time / 1000;
        long day = second / 86400;
        second = second % 86400;
        long hour = second / 3600;
        second = second % 3600;
        long minute = second / 60;
        second = second % 60;

        String duration = second + " 秒";
        if (minute > 0) {
            duration = minute + " 分钟 " + duration;
        }
        if (hour > 0) {
            duration = hour + " 小时 " + duration;
        }
        if (day > 0) {
            duration = day + " 天 " + duration;
        }
        return duration;
    }

    private static CommandLine parse(String[] arguments) throws ParseException {
        CommandLineParser parser = new DefaultParser();
        Options options = new Options();
        options.addOption("n", "name", true, "name of container");
        options.addOption("p", "port", true, "port of container");
        options.addOption("e", true, "environment of container");
        options.addOption("v", "volume", true, "volume of container");
        options.addOption("idt", false, "idt of container");
        options.addOption("", "restart", true, "restart of container");
        options.addOption("", "privileged", true, "privileged of container");
        return parser.parse(options, arguments);
    }

    @Override
    public void execute() throws MojoExecutionException {
        Instant now = Instant.now();
        Shell shell = new Shell(host, port, username, password);
        DockerUtils.instance(shell).deploy(startFile);

        String appName;
        int appPort;
        try {
            List<String> dockers = Files.readAllLines(Paths.get(startFile));
            Optional<String> dockerRunPort = dockers.stream().filter(p -> p.startsWith("PORT")).findFirst();
            Optional<String> dockerRunName = dockers.stream().filter(p -> p.startsWith("NAME")).findFirst();

            if (dockerRunPort.isPresent() && dockerRunName.isPresent()) {
                appPort = Integer.parseInt(dockerRunPort.get().replace("PORT=", ""));
                appName = dockerRunName.get().replace("NAME=", "").replace("-$PORT", "");
            } else {
                String dockerRun = dockers.stream().filter(p -> p.startsWith("docker run")).findFirst().get();

                CommandLine cli = parse(dockerRun.split(" "));

                String optionName = cli.getOptionValue("n");
                appPort = Integer.parseInt(cli.getOptionValue("p").split(":")[0]);
                appName = optionName.replace("-" + appPort, "").toLowerCase();
            }
        } catch (Exception e) {
            try {
                String middleName = imageName.replace("-api", "").replace("dapware", "");
                Middle middle = Middle.valueOf(middleName.toUpperCase());
                appName = middle.name().toLowerCase();
                appPort = middle.port();
            } catch (Exception ex) {
                getLog().info("无法确定具体中间件，请自行检查是否部署成功。");
                return;
            }
        }

        String uri = String.format("http://%s:%s", host, appPort);
        getLog().info("---------------------------[ health check ]-----------------------------");

        boolean areYouOK = DockerUtils.instance(shell).areYouOK(uri);

        getLog().info("------------------------------------------------------------------------");
        if (areYouOK) {
            long elapsed = System.currentTimeMillis() - now.toEpochMilli();

            getLog().info(String.format("%s:%s 部署成功, 共耗时：%s.", imageName, imageTag, toDuration(elapsed)));
            getLog().info(String.format("首页地址: %s", uri));
            String envUri = String.format("%s/api/%s/v2/env", uri, appName);
            getLog().info(String.format("环境地址: %s", envUri));
            String actuatorUri = String.format("%s/%s/actuator", uri, appName);
            getLog().info(String.format("监控地址: %s", actuatorUri));
        } else {
            getLog().info(String.format("%s:%s 部署失败. 请继续等待.", imageName, imageTag));
        }
        getLog().info("------------------------------------------------------------------------");
    }
}
