package com.digiwin.dap.middleware.mybatis;

import com.alibaba.druid.wall.Violation;
import com.alibaba.druid.wall.WallCheckResult;
import com.alibaba.druid.wall.WallConfig;
import com.alibaba.druid.wall.WallProvider;
import com.alibaba.druid.wall.spi.MySqlWallProvider;
import com.alibaba.druid.wall.violation.SyntaxErrorViolation;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;

import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * 防SQL注入拦截器
 *
 * @author fobgochod
 * @date 2023/3/15 13:32
 */
@Intercepts({
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}),
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class}),
})
public class InjectionInterceptor implements Interceptor {

    private static final WallProvider provider;
    private static final String ORDER_BY = "orderBy";
    /**
     * 仅支持字母、数字、下划线、空格、逗号、小数点（支持多个字段排序）
     */
    private static final String SQL_PATTERN = "[a-zA-Z0-9_\\ \\,\\.]+";

    static {
        WallConfig config = new WallConfig(MySqlWallProvider.DEFAULT_CONFIG_DIR);
        config.setSelectWhereAlwayTrueCheck(false);
        config.setConditionAndAlwayTrueAllow(true);
        provider = new MySqlWallProvider(config);
    }

    public void checkInjection(Object parameter, String sql) throws SQLException {
        if (parameter instanceof Map) {
            Map parameters = (Map) parameter;
            if (parameters.containsKey(ORDER_BY)) {
                String value = parameters.get(ORDER_BY) == null ? "" : parameters.get(ORDER_BY).toString();
                if (!value.isEmpty()) {
                    boolean matches = value.matches(SQL_PATTERN);
                    if (!matches) {
                        throw new SQLException("sql injection violation, " + value + " : " + sql);
                    }
                }
            }
        }

        WallCheckResult checkResult = provider.check(sql);
        List<Violation> violations = checkResult.getViolations();
        if (violations.size() > 0) {
            Violation firstViolation = violations.get(0);
            if (violations.get(0) instanceof SyntaxErrorViolation) {
                SyntaxErrorViolation violation = (SyntaxErrorViolation) violations.get(0);
                throw new SQLException("sql injection violation, " + firstViolation.getMessage() + " : " + sql, violation.getException());
            } else {
                throw new SQLException("sql injection violation, " + firstViolation.getMessage() + " : " + sql);
            }
        }
    }

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        Object[] args = invocation.getArgs();
        MappedStatement ms = (MappedStatement) args[0];
        Object parameter = args[1];
        RowBounds rowBounds = (RowBounds) args[2];
        ResultHandler resultHandler = (ResultHandler) args[3];
        Executor executor = (Executor) invocation.getTarget();
        CacheKey cacheKey;
        BoundSql boundSql;
        //由于逻辑关系，只会进入一次
        if (args.length == 4) {
            //4 个参数时
            boundSql = ms.getBoundSql(parameter);
            cacheKey = executor.createCacheKey(ms, parameter, rowBounds, boundSql);
        } else {
            //6 个参数时
            cacheKey = (CacheKey) args[4];
            boundSql = (BoundSql) args[5];
        }
        //sql注入检查
        checkInjection(parameter, boundSql.getSql());
        //注：下面的方法可以根据自己的逻辑调用多次，在分页插件中，count 和 page 各调用了一次
        return executor.query(ms, parameter, rowBounds, resultHandler, cacheKey, boundSql);
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }

    @Override
    public void setProperties(Properties properties) {
    }
}
