package com.digiwin.dap.middleware.mybatis.interceptor;

import com.digiwin.dap.middle.database.encrypt.desensitization.context.DesensitizationConvertContext;
import com.digiwin.dap.middle.database.encrypt.desensitization.service.DesensitizationConverter;
import com.digiwin.dap.middle.database.encrypt.model.ObjectRelationalMapping;
import com.digiwin.dap.middle.database.encrypt.sql.parser.SqlParser;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.plugin.Intercepts;
import org.apache.ibatis.plugin.Invocation;
import org.apache.ibatis.plugin.Signature;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Properties;

/**
 * @author michael
 * <p>mybatis拦截器,拦截所有select语句对结果集进行解密,需要注意一下三点：
 * <p>1.selec语句中不可出现select *</p>
 * <p>2.xml中如果配置了resultMap不支持解密,推荐使用自定义TypeHandler</p>
 * <p>3.结果集列上进行了函数操作不支持解密</p>
 * </p>
 */
@Intercepts({@Signature(
        type = Executor.class,
        method = "query",
        args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}
), @Signature(
        type = Executor.class,
        method = "query",
        args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class}
)})
public class DecryptResultInterceptor implements Interceptor {

    private final static Logger LOGGER = LoggerFactory.getLogger(DecryptResultInterceptor.class);

    private final DesensitizationConverter<Object> desensitizationConverter;

    private final SqlParser sqlParser;

    public DecryptResultInterceptor(DesensitizationConverter<Object> desensitizationConverter, SqlParser sqlParser) {
        this.desensitizationConverter = desensitizationConverter;
        this.sqlParser = sqlParser;
    }

    @Override
    public Object intercept(Invocation invocation) throws InvocationTargetException, IllegalAccessException {
        Object result = invocation.proceed();
        String targetMethod = "";
        try {
            // 判断xml返回是否是ResultMap,ResultMap使用TypeHandler处理
            MappedStatement mappedStatement = (MappedStatement) invocation.getArgs()[0];
            if (mappedStatement.getResultMaps() != null && !mappedStatement.getResultMaps().isEmpty()
                    && mappedStatement.getResultMaps().get(0).getResultMappings() != null
                    && !mappedStatement.getResultMaps().get(0).getResultMappings().isEmpty()) {
                return result;
            }
            Object[] args = invocation.getArgs();
            targetMethod = mappedStatement.getId();
            String sql = mappedStatement.getBoundSql(args[1]).getSql();

            List<ObjectRelationalMapping> objectRelationalMappingList = sqlParser.parseQuerySql(targetMethod, sql);

            DesensitizationConvertContext<Object> resultContext
                    = new DesensitizationConvertContext<>(targetMethod, objectRelationalMappingList);
            resultContext.setContext(result);
            return desensitizationConverter.revert(resultContext);
        } catch (Exception e) {
            String errorMsg = StringUtils.hasLength(targetMethod) ? targetMethod + "方法对结果集解密失败" : "mybatis解密结果集异常";
            LOGGER.error("===>{}", errorMsg, e);
        }
        return result;
    }

    @Override
    public Object plugin(Object target) {
        return Interceptor.super.plugin(target);
    }

    @Override
    public void setProperties(Properties properties) {
        Interceptor.super.setProperties(properties);
    }
}