package com.digiwin.dap.middle.autoconfigure.config;

import com.digiwin.dap.middle.autoconfigure.properties.RamProperties;
import com.digiwin.dap.middle.ram.constant.CacheConstant;
import com.digiwin.dap.middle.ram.util.JsonUtils;
import com.digiwin.dap.middle.ram.util.RedisUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;

/**
 * 配置redis的对象序列化器，把默认的变成json的序列化器
 *
 * @author zhouxiao
 * @date 2021/6/25
 */
@Configuration
@EnableCaching
@ConditionalOnClass(RedisConnectionFactory.class)
@EnableConfigurationProperties(RamProperties.class)
public class RamRedisConfiguration {

    @Value("${spring.application.name:dap}")
    private String appName;

    @Bean(CacheConstant.REDIS_TEMPLATE)
    public RedisTemplate<String, Object> innerRedisTemplate(RedisConnectionFactory factory) {
        RedisTemplate<String, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(factory);
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setValueSerializer(JsonUtils.getRedisSerializer());
        redisTemplate.afterPropertiesSet();

        RedisUtils.initTemplate(redisTemplate);
        return redisTemplate;
    }

    @Bean(CacheConstant.CACHE_MANAGER)
    public RedisCacheManager innerRedisCacheManager(RedisConnectionFactory factory) {
        RedisCacheWriter cacheWriter = RedisCacheWriter.nonLockingRedisCacheWriter(factory);
        RedisCacheConfiguration config = RedisCacheConfiguration.defaultCacheConfig()
                .computePrefixWith(cacheName -> appName + ":ram:" + cacheName + ":")
                .entryTtl(Duration.ofHours(2))
                .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(new StringRedisSerializer()))
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(JsonUtils.getRedisSerializer(true)))
                .disableCachingNullValues();
        return RedisCacheManager.builder(cacheWriter).cacheDefaults(config).build();
    }
}
