package com.digiwin.dap.middle.autoconfigure.config;

import com.digiwin.dap.middle.autoconfigure.properties.RamProperties;
import com.digiwin.dap.middle.ram.constant.Constant;
import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.filter.ResourceAccessManagementFilter;
import com.digiwin.dap.middle.ram.interceptor.ResourceAccessManagementInterceptor;
import com.digiwin.dap.middle.ram.mapper.RamPolicyMapper;
import com.digiwin.dap.middle.ram.service.PolicyQueryService;
import com.digiwin.dap.middle.ram.service.executor.GrantCachingExecutor;
import com.digiwin.dap.middle.ram.service.executor.GrantExecutor;
import com.digiwin.dap.middle.ram.service.executor.GrantSimpleExecutor;
import com.digiwin.dap.middle.ram.service.executor.PolicyCachingExecutor;
import com.digiwin.dap.middle.ram.service.executor.PolicyExecutor;
import com.digiwin.dap.middle.ram.service.executor.PolicySimpleExecutor;
import com.digiwin.dap.middle.ram.service.policy.filiter.AllowListPolicyFilterHandler;
import com.digiwin.dap.middle.ram.service.policy.filiter.BasePolicyFilterHandler;
import com.digiwin.dap.middle.ram.service.policy.filiter.BlockListPolicyFilterHandler;
import com.digiwin.dap.middle.ram.service.policy.filiter.NonePolicyFilterHandler;
import com.digiwin.dap.middle.ram.service.policy.filiter.PolicyFilterHandler;
import com.digiwin.dap.middle.ram.service.policy.interceptor.FunctionPolicyInterceptorHandler;
import com.digiwin.dap.middle.ram.service.policy.interceptor.ObsoletePolicyInterceptorHandler;
import com.digiwin.dap.middle.ram.service.policy.interceptor.PolicyInterceptorHandler;
import org.mybatis.spring.annotation.MapperScan;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotationAwareOrderComparator;
import org.springframework.web.servlet.handler.MappedInterceptor;

import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 包扫描
 *
 * @author Seven
 * @date 2021/8/20 11:45
 */
@Configuration
@ComponentScan(basePackages = "com.digiwin.dap.middle.ram")
@MapperScan("com.digiwin.dap.middle.ram.mapper")
@EnableConfigurationProperties(RamProperties.class)
public class RamScanConfiguration {

    private final RamProperties properties;

    public RamScanConfiguration(RamProperties properties) {
        this.properties = properties;
    }

    @Bean
    public GrantExecutor grantExecutor(RamPolicyMapper ramPolicyMapper) {
        GrantExecutor grantExecutor = new GrantSimpleExecutor(ramPolicyMapper);
        if (properties.isCache()) {
            grantExecutor = new GrantCachingExecutor(grantExecutor);
        }
        return grantExecutor;
    }

    @Bean
    public PolicyExecutor policyExecutor(RamPolicyMapper ramPolicyMapper) {
        PolicyExecutor policyExecutor = new PolicySimpleExecutor(ramPolicyMapper);
        if (properties.isCache()) {
            policyExecutor = new PolicyCachingExecutor(policyExecutor);
        }
        return policyExecutor;
    }

    @Bean
    @ConditionalOnBean(PolicyQueryService.class)
    public List<PolicyFilterHandler> policyFilterHandlers(PolicyQueryService policyQueryService) {
        List<PolicyFilterHandler> policyHandlers = new ArrayList<>();
        policyHandlers.add(new AllowListPolicyFilterHandler(policyQueryService));
        policyHandlers.add(new BlockListPolicyFilterHandler());
        policyHandlers.add(new BasePolicyFilterHandler(policyQueryService));
        policyHandlers.add(new NonePolicyFilterHandler(policyQueryService));
        AnnotationAwareOrderComparator.sort(policyHandlers);
        return policyHandlers;
    }

    @Bean
    @ConditionalOnBean(PolicyQueryService.class)
    public List<PolicyInterceptorHandler> policyInterceptorHandlers(PolicyQueryService policyQueryService) {
        List<PolicyInterceptorHandler> handlers = new ArrayList<>();
        Set<PolicyType> policies = properties.getPolicy();
        if (policies.contains(PolicyType.Function)) {
            handlers.add(new FunctionPolicyInterceptorHandler(policyQueryService));
        }
        if (policies.contains(PolicyType.Obsolete)) {
            handlers.add(new ObsoletePolicyInterceptorHandler(policyQueryService));
        }
        AnnotationAwareOrderComparator.sort(handlers);
        return handlers;
    }

    @Bean
    @ConditionalOnBean(PolicyQueryService.class)
    @ConditionalOnMissingBean(name = "ramFilter")
    public FilterRegistrationBean<Filter> ramFilter(PolicyQueryService policyQueryService) {
        FilterRegistrationBean<Filter> registrationBean = new FilterRegistrationBean<>();
        registrationBean.setFilter(new ResourceAccessManagementFilter(policyFilterHandlers(policyQueryService)));
        registrationBean.setOrder(-99);
        registrationBean.addUrlPatterns(Constant.DEFAULT_URL_MAPPINGS);
        registrationBean.setAsyncSupported(true);
        return registrationBean;
    }

    @Bean
    @ConditionalOnBean(PolicyQueryService.class)
    @ConditionalOnMissingBean(name = "ramInterceptor")
    public MappedInterceptor ramInterceptor(PolicyQueryService policyQueryService) {
        ResourceAccessManagementInterceptor interceptor = new ResourceAccessManagementInterceptor(policyInterceptorHandlers(policyQueryService));
        return new MappedInterceptor(Constant.DEFAULT_PATH_MAPPINGS, interceptor);
    }
}
