package com.digiwin.dap.middle.ram.domain.condition;

import org.springframework.lang.Nullable;
import org.springframework.web.servlet.mvc.condition.RequestCondition;

import java.util.Collection;
import java.util.Iterator;

/**
 * A base class for {@link RequestCondition} types providing implementations of
 * {@link #equals(Object)}, {@link #hashCode()}, and {@link #toString()}.
 *
 * @author fobgochod
 * @date 2022/5/25 14:00
 */
public abstract class AbstractCondition {

    /**
     * Indicates whether this condition is empty, i.e. whether or not it
     * contains any discrete items.
     *
     * @return {@code true} if empty; {@code false} otherwise
     */
    public boolean isEmpty() {
        return getContent().isEmpty();
    }

    /**
     * Return the discrete items a request condition is composed of.
     * <p>For example URL patterns, HTTP request methods, param expressions, etc.
     *
     * @return a collection of objects (never {@code null})
     */
    protected abstract Collection<?> getContent();

    /**
     * The notation to use when printing discrete items of content.
     * <p>For example {@code " || "} for URL patterns or {@code " && "}
     * for param expressions.
     */
    protected String getToStringInfix() {
        return " || ";
    }


    @Override
    public boolean equals(@Nullable Object other) {
        if (this == other) {
            return true;
        }
        if (other == null || getClass() != other.getClass()) {
            return false;
        }
        return getContent().equals(((AbstractCondition) other).getContent());
    }

    @Override
    public int hashCode() {
        return getContent().hashCode();
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder("[");
        for (Iterator<?> iterator = getContent().iterator(); iterator.hasNext(); ) {
            Object expression = iterator.next();
            builder.append(expression.toString());
            if (iterator.hasNext()) {
                builder.append(getToStringInfix());
            }
        }
        builder.append("]");
        return builder.toString();
    }
}
