package com.digiwin.dmc.sdk.service;

import com.digiwin.dmc.sdk.entity.BatchOperationIds;
import com.digiwin.dmc.sdk.entity.ContentsResult;
import com.digiwin.dmc.sdk.entity.DirInfo;
import com.digiwin.dmc.sdk.entity.FileInfo;
import com.digiwin.dmc.sdk.entity.Query.FileInfoQuery;
import com.digiwin.dmc.sdk.entity.Query.FullTextCondition;
import com.digiwin.dmc.sdk.entity.Query.QueryResult;
import com.digiwin.dmc.sdk.service.download.IGeneralDocumentDownloader;
import com.digiwin.dmc.sdk.service.download.ISegmentDocumentDownloader;
import com.digiwin.dmc.sdk.service.upload.ICoverGeneralDocumentUploader;
import com.digiwin.dmc.sdk.service.upload.ICoverSegmentDocumentUploader;
import com.digiwin.dmc.sdk.service.upload.IGeneralDocumentUploader;
import com.digiwin.dmc.sdk.service.upload.ISegmentDocumentUploader;

/**
 * @author chenzwd
 * @date : 2018-06-14 13:59
 * @Description:
 */
public interface IDocumentStorageService {

    /**
     * 获取普通文档上传器（默认存储区）(覆盖)
     *
     * @param localPath 上传文件路径
     * @param fileId    文件id
     * @return 普通文档上传器
     */
    ICoverGeneralDocumentUploader coverUploadDocument(String localPath, String fileId);

    ICoverGeneralDocumentUploader coverUploadDocument(byte[] bytes, String fileId);

    /**
     * 获取普通文档上传器（指定存储区）(覆盖)
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileId     文件id
     * @return 普通文档上传器
     */
    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, String localPath, String fileId);

    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, byte[] bytes, String fileId);

    /**
     * 获取普通文档上传器（跨库上传）(覆盖)
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileId     文件id
     * @param driveToken driveToken
     * @return 普通文档上传器
     */
    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, String localPath, String fileId, String driveToken);

    ICoverGeneralDocumentUploader coverUploadDocument(String bucketName, byte[] bytes, String fileId, String driveToken);

    /**
     * 获取分段文档上传器（默认存储区）(覆盖)
     *
     * @param localPath 路径
     * @param fileId    文件id
     * @return 普通文档上传器
     */
    ICoverSegmentDocumentUploader getUploader(String localPath, String fileId);

    ICoverSegmentDocumentUploader getUploader(byte[] bytes, String fileId);

    /**
     * 获取分段文档上传器（指定存储区）(覆盖)
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileId     文件id
     * @return 普通文档上传器
     */
    ICoverSegmentDocumentUploader getUploader(String bucketName, String localPath, String fileId);

    ICoverSegmentDocumentUploader getUploader(String bucketName, byte[] bytes, String fileId);

    /**
     * 获取分段文档上传器（指定存储区）(覆盖)
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileId     文件id
     * @return 普通文档上传器
     */
    ICoverSegmentDocumentUploader getUploader(String bucketName, String localPath, String fileId, String driveToken);

    ICoverSegmentDocumentUploader getUploader(String bucketName, byte[] bytes, String fileId, String driveToken);

    /**
     * 获取普通文档上传器（默认存储区）
     *
     * @param localPath 上传文件路径
     * @param fileInfo  文件信息
     * @return 普通文档上传器
     */
    IGeneralDocumentUploader uploadDocument(String localPath, FileInfo fileInfo);

    IGeneralDocumentUploader uploadDocument(byte[] bytes, FileInfo fileInfo);

    /**
     * 获取普通文档上传器（指定存储区）
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileInfo   文件信息
     * @return 普通文档上传器
     */
    IGeneralDocumentUploader uploadDocument(String bucketName, String localPath, FileInfo fileInfo);

    IGeneralDocumentUploader uploadDocument(String bucketName, byte[] bytes, FileInfo fileInfo);

    /**
     * 获取普通文档上传器（跨库上传）
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileInfo   文件信息
     * @param driveToken driveToken
     * @return 普通文档上传器
     */
    IGeneralDocumentUploader uploadDocument(String bucketName, String localPath, FileInfo fileInfo, String driveToken);

    IGeneralDocumentUploader uploadDocument(String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken);


    /**
     * 获取分段文档上传器（默认存储区）
     *
     * @param localPath 上传文件路径
     * @param fileInfo  文件信息
     * @return 普通文档上传器
     */
    ISegmentDocumentUploader getUploader(String localPath, FileInfo fileInfo);

    ISegmentDocumentUploader getUploader(byte[] bytes, FileInfo fileInfo);

    /**
     * 获取分段文档上传器（指定存储区）
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileInfo   文件信息
     * @return 普通文档上传器
     */
    ISegmentDocumentUploader getUploader(String bucketName, String localPath, FileInfo fileInfo);

    ISegmentDocumentUploader getUploader(String bucketName, byte[] bytes, FileInfo fileInfo);

    /**
     * 获取分段文档上传器（指定存储区）
     *
     * @param bucketName 存储区名称
     * @param localPath  上传文件路径
     * @param fileInfo   文件信息
     * @return 普通文档上传器
     */
    ISegmentDocumentUploader getUploader(String bucketName, String localPath, FileInfo fileInfo, String driveToken);

    ISegmentDocumentUploader getUploader(String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken);


    /**
     * 获取普通文档下载器
     *
     * @param fileId
     * @param saveToPath
     */
    IGeneralDocumentDownloader downloadDocument(String fileId, String saveToPath);

    IGeneralDocumentDownloader downloadDocument(String bucketName, String fileId, String saveToPath);

    IGeneralDocumentDownloader downloadDocument(String bucketName, String fileId, String saveToPath, String driveToken);


    /**
     * 获取分段文档下载器
     *
     * @param fileId
     * @param saveToPath
     * @return
     */
    ISegmentDocumentDownloader getDownloader(String fileId, String saveToPath);

    ISegmentDocumentDownloader getDownloader(String bucketName, String fileId, String saveToPath);

    ISegmentDocumentDownloader getDownloader(String bucketName, String fileId, String saveToPath, String driveToken);


    /**
     * 删除文档
     *
     * @param fileId
     * @return String
     */
    String deleteDocument(String fileId);

    String deleteDocument(String bucketName, String fileId);

    String deleteDocument(String bucketName, String fileId, String driveToken);


    /**
     * 批量删除
     *
     * @param operationIds
     */
    void deleteManyDocument(BatchOperationIds operationIds);

    void deleteManyDocument(String bucketName, BatchOperationIds operationIds);

    void deleteManyDocument(String bucketName, BatchOperationIds operationIds, String driveToken);


    /**
     * 获取文档信息
     *
     * @param fileId
     * @return
     */
    FileInfo getDocumentInfo(String fileId);

    FileInfo getDocumentInfo(String bucketName, String fileId);

    FileInfo getDocumentInfo(String bucketName, String fileId, String driveToken);


    /**
     * 重命名文件
     *
     * @param fileId
     * @param fileName
     */
    void updateDocument(String fileId, String fileName);

    void updateDocument(String bucketName, String fileId, String fileName);

    void updateDocument(String bucketName, String fileId, String fileName, String driveToken);

    /**
     * 移动文档
     *
     * @param fileId
     * @param targetDirectoryId
     */
    void moveDocument(String fileId, String targetDirectoryId);

    void moveDocument(String bucketName, String fileId, String targetDirectoryId);

    void moveDocument(String bucketName, String fileId, String targetDirectoryId, String driveToken);


    /**
     * 批量移动
     *
     * @param operationIds
     * @param targetDirectoryId
     */
    void moveManyDocument(BatchOperationIds operationIds, String targetDirectoryId);

    void moveManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId);

    void moveManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId, String driveToken);

    /**
     * 复制文档
     *
     * @param fileId
     * @param targetDirectoryId
     * @return
     */
    String copyDocument(String fileId, String targetDirectoryId);

    String copyDocument(String bucketName, String fileId, String targetDirectoryId);

    String copyDocument(String bucketName, String fileId, String targetDirectoryId, String driveToken);

    /**
     * 批量复制
     *
     * @param operationIds
     * @param targetDirectoryId
     */
    void copyManyDocument(BatchOperationIds operationIds, String targetDirectoryId);

    void copyManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId);

    void copyManyDocument(String bucketName, BatchOperationIds operationIds, String targetDirectoryId, String driveToken);


    /**
     * 创建目录
     *
     * @param directoryName
     * @param parentDirectoryId
     * @return
     */
    String createDirectory(String directoryName, String parentDirectoryId);

    String createDirectory(String bucketName, String directoryName, String parentDirectoryId);

    String createDirectory(String bucketName, String directoryName, String parentDirectoryId, String driveToken);

    /**
     * 获取目录信息
     *
     * @param directoryId
     * @return
     */
    DirInfo getDirectoryInfo(String directoryId);

    DirInfo getDirectoryInfo(String bucketName, String directoryId);

    DirInfo getDirectoryInfo(String bucketName, String directoryId, String driveToken);

    /**
     * 删除目录
     *
     * @param directoryId
     */
    String deleteDirectory(String directoryId);

    String deleteDirectory(String bucketName, String directoryId);

    String deleteDirectory(String bucketName, String directoryId, String driveToken);


    /**
     * 更新目录名称
     *
     * @param directoryId
     * @param directoryName
     */
    void updateDirectory(String directoryId, String directoryName);

    void updateDirectory(String bucketName, String directoryId, String directoryName);

    void updateDirectory(String bucketName, String directoryId, String directoryName, String driveToken);

    /**
     * 移动目录
     *
     * @param directoryId
     * @param targetDirectoryId
     */
    void moveDirectory(String directoryId, String targetDirectoryId);

    void moveDirectory(String bucketName, String directoryId, String targetDirectoryId);

    void moveDirectory(String bucketName, String directoryId, String targetDirectoryId, String driveToken);

    /**
     * 复制目录
     *
     * @param directoryId
     * @param targetDirectoryId
     * @return
     */
    String copyDirectory(String directoryId, String targetDirectoryId);

    String copyDirectory(String bucketName, String directoryId, String targetDirectoryId);

    String copyDirectory(String bucketName, String directoryId, String targetDirectoryId, String driveToken);

    /**
     * 列出目录下的文件和子目录
     *
     * @param directoryId
     * @return
     */
    ContentsResult listContents(String directoryId);

    ContentsResult listContents(String bucketName, String directoryId);

    ContentsResult listContents(String bucketName, String directoryId, String driveToken);


    /**
     * 根据FileInfoQuery查询文档
     *
     * @param bucketName 存储区
     * @param query      查询条件
     * @param driveToken driveToken
     * @return 查询结果
     */
    QueryResult queryFileInfoByFileInfoQuery(String bucketName, FileInfoQuery query, String driveToken);

    QueryResult queryFileInfoByFileInfoQuery(String bucketName, FileInfoQuery query);

    QueryResult queryFileInfoByFileInfoQuery(FileInfoQuery query);


    /**
     * 根据单个字段精确查询
     *
     * @param bucketName 存储区
     * @param field      字段名
     * @param value      字段值
     * @param driveToken driveToken
     * @return 查询结果
     */
    QueryResult queryFileInfoByField(String bucketName, String field, String value, String driveToken);

    QueryResult queryFileInfoByField(String bucketName, String field, String value);

    QueryResult queryFileInfoByField(String field, String value);

    /**
     * 根据单个字段精确查询
     *
     * @param bucketName 存储区
     * @param field      字段名
     * @param value      字段值
     * @param pageNum    当前页
     * @param pageSize   显示个数
     * @param driveToken driveToken
     * @return 查询结果
     */
    QueryResult queryFileInfoByFieldWithPage(String bucketName, String field, String value, String pageNum, String pageSize, String driveToken);

    QueryResult queryFileInfoByFieldWithPage(String bucketName, String field, String value, String pageNum, String pageSize);

    QueryResult queryFileInfoByFieldWithPage(String field, String value, String pageNum, String pageSize);


    /**
     * 单个字段比较查询
     *
     * @param bucketName 存储区
     * @param field      字段名
     * @param operator   操作符
     * @param value      字段值
     * @param pageNum    当前页
     * @param pageSize   显示个数
     * @param driveToken driveToken
     * @return 查询结果
     */
    QueryResult queryFileInfoByOperatorWithPage(String bucketName, String field, String operator, String value, String pageNum, String pageSize, String driveToken);

    QueryResult queryFileInfoByOperatorWithPage(String bucketName, String field, String operator, String value, String pageNum, String pageSize);

    QueryResult queryFileInfoByOperatorWithPage(String field, String operator, String value, String pageNum, String pageSize);

    /**
     * 单个字段比较查询
     *
     * @param bucketName 存储区
     * @param field      字段名
     * @param operator   操作符
     * @param value      字段值
     * @param driveToken driveToken
     * @return 查询结果
     */
    QueryResult queryFileInfoByOperator(String bucketName, String field, String operator, String value, String driveToken);

    QueryResult queryFileInfoByOperator(String bucketName, String field, String operator, String value);

    QueryResult queryFileInfoByOperator(String field, String operator, String value);

    /**
     * 全文搜索，针对一个库
     *
     * @param bucketName 存储区
     * @param search     搜索条件
     * @param driveToken driveToken
     * @return 查询结果
     */
    QueryResult queryFileInfoByFullText(String bucketName, FullTextCondition search, String driveToken);

    QueryResult queryFileInfoByFullText(String bucketName, FullTextCondition search);

    QueryResult queryFileInfoByFullText(FullTextCondition search);


    /**
     * 全文搜索，针对多个库
     *
     * @param search      搜索条件
     * @param accessToken 操作Token
     * @return
     */
    QueryResult queryFileInfoByFullText(FullTextCondition search, String accessToken);


    /**
     * <ol>
     *      <li>以下API是旧版本Bucket租户拼接方案，此方案已经废弃，请不要使用.</li>
     *      <li>给文件添加租户ID，可以通过如下方案：</li>
     *      <ol>
     *          <li>旧版API(即：com.digiwin.dmc)：上传文件，通过{@link com.digiwin.dmc.sdk.entity.FileInfo}设置tenantId的值即可</li>
     *          <li>新版API(即：com.digiwin.dap.middleware)：{@link  com.digiwin.dap.middleware.dmc.DMC}</li>
     *      </ol>
     *      <li>新版本使用范例：<a href="http://10.40.42.38/middleware/dapware-example/tree/master/java-example/src/main/java/com/digiwin/dap/sdk/dmc">参考 </a></li>
     * <ol>
     **/
    @Deprecated
    IGeneralDocumentUploader uploadDocumentByTenant(String tenantId, String bucketName, String localPath, FileInfo fileInfo);

    @Deprecated
    IGeneralDocumentUploader uploadDocumentByTenant(String tenantId, String bucketName, byte[] bytes, FileInfo fileInfo);

    @Deprecated
    IGeneralDocumentUploader uploadDocumentByTenant(String tenantId, String bucketName, String localPath, FileInfo fileInfo, String driveToken);

    @Deprecated
    IGeneralDocumentUploader uploadDocumentByTenant(String tenantId, String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken);

    @Deprecated
    ISegmentDocumentUploader getUploaderByTenant(String tenantId, String localPath, FileInfo fileInfo);

    @Deprecated
    ISegmentDocumentUploader getUploaderByTenant(String tenantId, byte[] bytes, FileInfo fileInfo);

    @Deprecated
    ISegmentDocumentUploader getUploaderByTenant(String tenantId, String bucketName, String localPath, FileInfo fileInfo);

    @Deprecated
    ISegmentDocumentUploader getUploaderByTenant(String tenantId, String bucketName, byte[] bytes, FileInfo fileInfo);

    @Deprecated
    ISegmentDocumentUploader getUploaderByTenant(String tenantId, String bucketName, String localPath, FileInfo fileInfo, String driveToken);

    @Deprecated
    ISegmentDocumentUploader getUploaderByTenant(String tenantId, String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken);

    @Deprecated
    IGeneralDocumentDownloader downloadDocumentByTenant(String tenantId, String fileId, String saveToPath);

    @Deprecated
    IGeneralDocumentDownloader downloadDocumentByTenant(String tenantId, String bucketName, String fileId, String saveToPath);

    @Deprecated
    IGeneralDocumentDownloader downloadDocumentByTenant(String tenantId, String bucketName, String fileId, String saveToPath, String driveToken);

    @Deprecated
    ISegmentDocumentDownloader getDownloaderByTenant(String tenantId, String fileId, String saveToPath);

    @Deprecated
    ISegmentDocumentDownloader getDownloaderByTenant(String tenantId, String bucketName, String fileId, String saveToPath);

    @Deprecated
    ISegmentDocumentDownloader getDownloaderByTenant(String tenantId, String bucketName, String fileId, String saveToPath, String driveToken);

    @Deprecated
    String deleteDocumentByTenant(String tenantId, String fileId);

    @Deprecated
    String deleteDocumentByTenant(String tenantId, String bucketName, String fileId);

    @Deprecated
    String deleteDocumentByTenant(String tenantId, String bucketName, String fileId, String driveToken);

    @Deprecated
    void deleteManyDocumentByTenant(String tenantId, BatchOperationIds operationIds);

    @Deprecated
    void deleteManyDocumentByTenant(String tenantId, String bucketName, BatchOperationIds operationIds);

    @Deprecated
    void deleteManyDocumentByTenant(String tenantId, String bucketName, BatchOperationIds operationIds, String driveToken);

    @Deprecated
    FileInfo getDocumentInfoByTenant(String tenantId, String fileId);

    @Deprecated
    FileInfo getDocumentInfoByTenant(String tenantId, String bucketName, String fileId);

    @Deprecated
    FileInfo getDocumentInfoByTenant(String tenantId, String bucketName, String fileId, String driveToken);

    @Deprecated
    void updateDocumentByTenant(String tenantId, String fileId, String fileName);

    @Deprecated
    void updateDocumentByTenant(String tenantId, String bucketName, String fileId, String fileName);

    @Deprecated
    void updateDocumentByTenant(String tenantId, String bucketName, String fileId, String fileName, String driveToken);

    @Deprecated
    void moveDocumentByTenant(String tenantId, String fileId, String targetDirectoryId);

    @Deprecated
    void moveDocumentByTenant(String tenantId, String bucketName, String fileId, String targetDirectoryId);

    @Deprecated
    void moveDocumentByTenant(String tenantId, String bucketName, String fileId, String targetDirectoryId, String driveToken);

    @Deprecated
    void moveManyDocumentByTenant(String tenantId, BatchOperationIds operationIds, String targetDirectoryId);

    @Deprecated
    void moveManyDocumentByTenant(String tenantId, String bucketName, BatchOperationIds operationIds, String targetDirectoryId);

    @Deprecated
    void moveManyDocumentByTenant(String tenantId, String bucketName, BatchOperationIds operationIds, String targetDirectoryId, String driveToken);

    @Deprecated
    String copyDocumentByTenant(String tenantId, String fileId, String targetDirectoryId);

    @Deprecated
    String copyDocumentByTenant(String tenantId, String bucketName, String fileId, String targetDirectoryId);

    @Deprecated
    String copyDocumentByTenant(String tenantId, String bucketName, String fileId, String targetDirectoryId, String driveToken);

    @Deprecated
    void copyManyDocumentByTenant(String tenantId, BatchOperationIds operationIds, String targetDirectoryId);

    @Deprecated
    void copyManyDocumentByTenant(String tenantId, String bucketName, BatchOperationIds operationIds, String targetDirectoryId);

    @Deprecated
    void copyManyDocumentByTenant(String tenantId, String bucketName, BatchOperationIds operationIds, String targetDirectoryId, String driveToken);

    @Deprecated
    String createDirectoryByTenant(String tenantId, String directoryName, String parentDirectoryId);

    @Deprecated
    String createDirectoryByTenant(String tenantId, String bucketName, String directoryName, String parentDirectoryId);

    @Deprecated
    String createDirectoryByTenant(String tenantId, String bucketName, String directoryName, String parentDirectoryId, String driveToken);

    @Deprecated
    DirInfo getDirectoryInfoByTenant(String tenantId, String directoryId);

    @Deprecated
    DirInfo getDirectoryInfoByTenant(String tenantId, String bucketName, String directoryId);

    @Deprecated
    DirInfo getDirectoryInfoByTenant(String tenantId, String bucketName, String directoryId, String driveToken);

    @Deprecated
    String deleteDirectoryByTenant(String tenantId, String directoryId);

    @Deprecated
    String deleteDirectoryByTenant(String tenantId, String bucketName, String directoryId);

    @Deprecated
    String deleteDirectoryByTenant(String tenantId, String bucketName, String directoryId, String driveToken);

    @Deprecated
    void updateDirectoryByTenant(String tenantId, String directoryId, String directoryName);

    @Deprecated
    void updateDirectoryByTenant(String tenantId, String bucketName, String directoryId, String directoryName);

    @Deprecated
    void updateDirectoryByTenant(String tenantId, String bucketName, String directoryId, String directoryName, String driveToken);

    @Deprecated
    void moveDirectoryByTenant(String tenantId, String directoryId, String targetDirectoryId);

    @Deprecated
    void moveDirectoryByTenant(String tenantId, String bucketName, String directoryId, String targetDirectoryId);

    @Deprecated
    void moveDirectoryByTenant(String tenantId, String bucketName, String directoryId, String targetDirectoryId, String driveToken);

    @Deprecated
    String copyDirectoryByTenant(String tenantId, String directoryId, String targetDirectoryId);

    @Deprecated
    String copyDirectoryByTenant(String tenantId, String bucketName, String directoryId, String targetDirectoryId);

    @Deprecated
    String copyDirectoryByTenant(String tenantId, String bucketName, String directoryId, String targetDirectoryId, String driveToken);

    @Deprecated
    ContentsResult listContentsByTenant(String tenantId, String directoryId);

    @Deprecated
    ContentsResult listContentsByTenant(String tenantId, String bucketName, String directoryId);

    @Deprecated
    ContentsResult listContentsByTenant(String tenantId, String bucketName, String directoryId, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByFileInfoQueryByTenant(String tenantId, String bucketName, FileInfoQuery query, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByFileInfoQueryByTenant(String tenantId, String bucketName, FileInfoQuery query);

    @Deprecated
    QueryResult queryFileInfoByFileInfoQueryByTenant(String tenantId, FileInfoQuery query);

    @Deprecated
    QueryResult queryFileInfoByFieldByTenant(String tenantId, String bucketName, String field, String value, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByFieldByTenant(String tenantId, String bucketName, String field, String value);

    @Deprecated
    QueryResult queryFileInfoByFieldByTenant(String tenantId, String field, String value);

    @Deprecated
    QueryResult queryFileInfoByFieldWithPageByTenant(String tenantId, String bucketName, String field, String value, String pageNum, String pageSize, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByFieldWithPageByTenant(String tenantId, String bucketName, String field, String value, String pageNum, String pageSize);

    @Deprecated
    QueryResult queryFileInfoByFieldWithPageByTenant(String tenantId, String field, String value, String pageNum, String pageSize);

    @Deprecated
    QueryResult queryFileInfoByOperatorWithPageByTenant(String tenantId, String bucketName, String field, String operator, String value, String pageNum, String pageSize, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByOperatorWithPageByTenant(String tenantId, String bucketName, String field, String operator, String value, String pageNum, String pageSize);

    @Deprecated
    QueryResult queryFileInfoByOperatorWithPageByTenant(String tenantId, String field, String operator, String value, String pageNum, String pageSize);

    @Deprecated
    QueryResult queryFileInfoByOperatorByTenant(String tenantId, String bucketName, String field, String operator, String value, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByOperatorByTenant(String tenantId, String bucketName, String field, String operator, String value);

    @Deprecated
    QueryResult queryFileInfoByOperatorByTenant(String tenantId, String field, String operator, String value);

    @Deprecated
    QueryResult queryFileInfoByFullTextByTenant(String tenantId, String bucketName, FullTextCondition search, String driveToken);

    @Deprecated
    QueryResult queryFileInfoByFullTextByTenant(String tenantId, String bucketName, FullTextCondition search);

    @Deprecated
    QueryResult queryFileInfoByFullTextByTenant(String tenantId, FullTextCondition search);

    /**
     * 获取普通文档上传器（默认存储区）
     *
     * @param tenantId  租户Id
     * @param localPath 上传文件路径
     * @param fileInfo  文件信息
     * @return 普通文档上传器
     */
    @Deprecated
    IGeneralDocumentUploader uploadDocumentByTenant(String tenantId, String localPath, FileInfo fileInfo);

    @Deprecated
    IGeneralDocumentUploader uploadDocumentByTenant(String tenantId, byte[] bytes, FileInfo fileInfo);
}
