package com.digiwin.dmc.sdk.service.download;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.NetworkOptions;
import com.digiwin.dmc.sdk.config.SDKConstants;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.service.impl.UserManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;


/**
 * @author chenzwd
 * @date : 2018-06-30 16:03
 * @Description:
 */
public class SegmentSegmentDocumentDownloader implements ISegmentDocumentDownloader, Runnable {

    private static final IUserManagerService userManagerService = UserManagerService.userInstance();
    private static final ThreadFactory threadFactory = new DefaultThreadFactory();

    private final Thread thread = threadFactory.newThread(this);
    private final String tenantId;
    private final String bucketName;
    private final String fileId;
    private final String driveToken;
    private final int bufferSize = NetworkOptions.Default.getPackageSize();
    private final DownloadProgressEventArgs eventArgs = new DownloadProgressEventArgs();
    private volatile DownloadProcessState state;
    private FileOutputStream fileOutputStream;
    private IDownLoadCallbackable completeCallback;
    private IDownLoadCallbackable progressChangedCallback;

    public SegmentSegmentDocumentDownloader(String tenantId, String bucketName, String fileId, String saveToPath, int fileSize, String driveToken) {
        try {
            File file = new File(saveToPath);
            if (!file.exists()) {
                file.createNewFile();
            }
            this.fileOutputStream = new FileOutputStream(file);
        } catch (Exception e) {
            throw new OperateException(e);
        }
        this.state = DownloadProcessState.NotStarted;
        this.tenantId = tenantId;
        this.bucketName = bucketName;
        this.fileId = fileId;
        this.driveToken = driveToken;
        this.eventArgs.setTotalBytes(fileSize);
    }

    /**
     * 异步下载文件内容
     */
    @Override
    public ISegmentDocumentDownloader beginDownload() {
        if (state != DownloadProcessState.NotStarted) {
            throw new OperateException("下载已开始，无法再次启动下载，请开启新的下载");
        }
        this.continueDownload();
        thread.start();
        return this;
    }

    @Override
    public void run() {
        this.multipartDownload(bucketName);
    }

    private void multipartDownload(String bucketName) {
        try {
            int totalBytes = eventArgs.getTotalBytes();
            int downloadBytes = 0;
            while (downloadBytes < totalBytes) {
                if (state == DownloadProcessState.Stopped) {
                    return;
                } else if (state == DownloadProcessState.Paused) {
                    LockSupport.park(thread);
                } else {
                    //from的位置是二进制数组的下标，所有需要把长度-1
                    int from = downloadBytes;
                    if (downloadBytes + bufferSize >= totalBytes) {
                        this.download(bucketName, from, totalBytes - downloadBytes);
                        downloadBytes = totalBytes;
                        eventArgs.setCompletedBytes(totalBytes);
                        eventArgs.setPercentage(1);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                        if (completeCallback != null) {
                            completeCallback.callback(eventArgs);
                        }
                    } else {
                        this.download(bucketName, from, bufferSize);
                        downloadBytes += bufferSize;
                        eventArgs.setCompletedBytes(downloadBytes);
                        eventArgs.setPercentage(Math.round(eventArgs.getCompletedBytes() * 10000.0 / totalBytes) / 10000.0);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    private void download(String bucketName, int bytesReaded, int count) {
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        String downloadResourceUrl = DmcUrl.getFilesResourceUrl() + bucketName + "/files/" + fileId + "/" + bytesReaded + "/" + count;
        try {
            Map<String, String> headers = new HashMap<>();
            if (StringUtil.isEmpty(driveToken)) {
                String userToken = userManagerService.getUserToken(ServerSetting.getUser());
                headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
            } else {
                headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
            }
            if (tenantId != null) {
                headers.put(SDKConstants.HTTP_HEADER_TENANTID_KEY, tenantId);
            }
            this.fileOutputStream = HttpRequestUtil.downloadSegFile(downloadResourceUrl, headers, count, fileOutputStream);
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    @Override
    public ISegmentDocumentDownloader onProgressChanged(IDownLoadCallbackable callbackable) {
        this.progressChangedCallback = callbackable;
        return this;
    }

    @Override
    public ISegmentDocumentDownloader onCompleted(IDownLoadCallbackable callbackable) {
        this.completeCallback = callbackable;
        return this;
    }

    @Override
    public void pauseDownload() {
        state = DownloadProcessState.Paused;
    }

    @Override
    public void continueDownload() {
        if (state == DownloadProcessState.Paused) {
            LockSupport.unpark(thread);
        }
        state = DownloadProcessState.Downloading;
    }

    @Override
    public void stopDownload() {
        state = DownloadProcessState.Stopped;
    }

    static class DefaultThreadFactory implements ThreadFactory {
        private final AtomicInteger threadNumber = new AtomicInteger(1);

        public Thread newThread(Runnable r) {
            return new Thread(r, "dmc-part-download-thread-" + threadNumber.getAndIncrement());
        }
    }
}
