package com.digiwin.dmc.sdk.service.impl;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.SDKConstants;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.BucketInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IBucketManagerService;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.ObjectMapperUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class BucketManagerService implements IBucketManagerService {

    private static BucketManagerService bucketManagerService;

    private static IUserManagerService userManagerService = UserManagerService.userInstance();

    /**
     * 单一实例
     *
     * @return
     */
    public static IBucketManagerService bucketInstance() {
        if (bucketManagerService == null) {
            bucketManagerService = new BucketManagerService();
        }
        return bucketManagerService;
    }

    /**
     * 创建存储区
     *
     * @param bucketInfo
     * @return
     */
    @Override
    public String createBucket(BucketInfo bucketInfo) {
        return createBucketByTenant(null, bucketInfo);
    }

    /**
     * 创建存储区-多租户
     *
     * @param bucketInfo
     * @return
     */
    @Override
    public String createBucketByTenant(String tenantId, BucketInfo bucketInfo) {
        if (bucketInfo == null) {
            throw new IllegalArgumentException("参数bucketInfo is null");
        }
        if (StringUtil.checkSpecialId(bucketInfo.getName())) {
            throw new IllegalArgumentException("bucketName为空或包含特殊字符.号！");
        }
        String url = DmcUrl.getBucketResourceUrl();
        String userToken = userManagerService.getUserToken(ServerSetting.getUser());
        Map<String, String> bucketMap = new HashMap<>();
        bucketMap.put("name", bucketInfo.getName());
        bucketMap.put("keytype", bucketInfo.getKeyType().toLowerCase());
        bucketMap.put("description", bucketInfo.getDescription());
        try {
            Map<String, String> headers = new HashMap<>();
            headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
            if (!StringUtil.isEmpty(tenantId)) {
                headers.put(SDKConstants.HTTP_HEADER_TENANTID_KEY, tenantId);
            }
            HashMap responseEntity = HttpRequestUtil.postJson(url,
                    ObjectMapperUtil.writeValueAsString(bucketMap), headers, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("id")) {
                return null;
            } else {
                return responseEntity.get("id").toString();
            }
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 查询存储区
     *
     * @return
     */
    @Override
    public List<BucketInfo> getBuckets() {
        String url = DmcUrl.getBucketResourceUrl();
        String userToken = userManagerService.getUserToken(ServerSetting.getUser());
        Map<String, String> headers = new HashMap<>();
        headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        ArrayList buckets = HttpRequestUtil.get(url, headers, ArrayList.class);
        List<BucketInfo> bucketInfos = new ArrayList<>();
        if (buckets != null) {
            for (Object bucket : buckets) {
                try {
                    String json = ObjectMapperUtil.writeValueAsString(bucket);
                    BucketInfo bucketInfo = ObjectMapperUtil.readValue(json, BucketInfo.class);
                    bucketInfos.add(bucketInfo);
                } catch (IOException e) {
                    throw new OperateException(e);
                }
            }
        }
        return bucketInfos;
    }

    /**
     * 判断存储区是否存在
     *
     * @return
     */
    @Override
    public String isExistBucket(String bucketName) {
        return isExistBucketByTenant(null, bucketName);
    }

    /**
     * 判断存储区是否存在-多租户
     *
     * @return
     */
    @Override
    public String isExistBucketByTenant(String tenantId, String bucketName) {
        if (bucketName == null || ("".equals(bucketName))) {
            throw new IllegalArgumentException("参数bucketName is null");
        }
        String url = DmcUrl.getBucketResourceUrl() + "/" + bucketName + "/isExist";
        String userToken = userManagerService.getUserToken(ServerSetting.getAdmin());
        Map<String, String> headers = new HashMap<>();
        headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        if (!StringUtil.isEmpty(tenantId)) {
            headers.put(SDKConstants.HTTP_HEADER_TENANTID_KEY, tenantId);
        }
        HashMap isExist = HttpRequestUtil.get(url, headers, HashMap.class);
        if (isExist == null || !isExist.containsKey("isExist")) {
            return null;
        }
        return isExist.get("isExist").toString();
    }

}
