package com.digiwin.dmc.sdk.service.impl;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.SDKConstants;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.FileInfo;
import com.digiwin.dmc.sdk.entity.SharedFile;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IShareFileService;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.service.upload.IShareUploader;
import com.digiwin.dmc.sdk.service.upload.ShareUploader;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.ObjectMapperUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class ShareFileService implements IShareFileService {

    private static IShareFileService shareFileService;
    private static IUserManagerService userManagerService = UserManagerService.userInstance();

    /**
     * 单一实例
     *
     * @return
     */
    public static IShareFileService shareFileInstance() {
        if (shareFileService == null) {
            shareFileService = new ShareFileService();
        }
        return shareFileService;
    }

    /**
     * 文件永久分享给所有人
     *
     * @param bucketName
     * @param fileIds
     * @param driveToken
     * @return
     */
    @Override
    public List<String> perpetualShareFiles(String bucketName, List<String> fileIds, String driveToken) {
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        if (fileIds == null || fileIds.size() == 0) {
            throw new IllegalArgumentException("参数fileId is null");
        }
        String shareFileUrl = DmcUrl.getFilesResourceUrl()
                + bucketName + "/ShareFiles";

        Map<String, String> headers = new HashMap<>();
        if (driveToken == null || "".equals(driveToken)) {
            String userToken = userManagerService.getUserToken(ServerSetting.getUser());
            headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        } else {
            headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
        }
        try {
            return HttpRequestUtil.postJson(shareFileUrl, ObjectMapperUtil.writeValueAsString(fileIds), headers, ArrayList.class);
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    @Override
    public List<String> perpetualShareFiles(String bucketName, List<String> fileIds) {
        return this.perpetualShareFiles(bucketName, fileIds, null);
    }

    @Override
    public List<String> perpetualShareFiles(List<String> fileIds) {
        return this.perpetualShareFiles(null, fileIds, null);
    }

    /**
     * 文件永久分享给所有人
     *
     * @param bucketName
     * @param fileIds
     * @param driveToken
     * @return
     */
    @Override
    public List<String> limitShareFiles(String bucketName, List<String> fileIds, int day, String driveToken) {
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        if (fileIds == null || fileIds.size() == 0) {
            throw new IllegalArgumentException("参数fileId is null");
        }
        String shareFileUrl = DmcUrl.getFilesResourceUrl()
                + bucketName + "/ShareFiles/d/" + day;

        Map<String, String> headers = new HashMap<>();
        if (driveToken == null || "".equals(driveToken)) {
            String userToken = userManagerService.getUserToken(ServerSetting.getUser());
            headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        } else {
            headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
        }
        try {
            return HttpRequestUtil.postJson(shareFileUrl, ObjectMapperUtil.writeValueAsString(fileIds), headers, ArrayList.class);
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    @Override
    public List<String> limitShareFiles(String bucketName, List<String> fileIds, int day) {
        return this.limitShareFiles(bucketName, fileIds, day, null);
    }

    @Override
    public List<String> limitShareFiles(List<String> fileIds, int day) {
        return this.limitShareFiles(null, fileIds, day, null);
    }

    /**
     * 取消分享
     *
     * @param bucketName
     * @param shareFileId
     * @param driveToken
     */
    @Override
    public void deleteShareFile(String bucketName, String shareFileId, String driveToken) {
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        if ((shareFileId == null) || ("".equals(shareFileId))) {
            throw new IllegalArgumentException("参数fileId is null");
        }
        String deleteShareFileUrl = DmcUrl.getResourceUrl() + "/"
                + bucketName + "/deleteShareFile/" + shareFileId;

        Map<String, String> headers = new HashMap<>();
        if (driveToken == null || "".equals(driveToken)) {
            String userToken = userManagerService.getUserToken(ServerSetting.getUser());
            headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        } else {
            headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
        }
        try {
            HttpRequestUtil.delete(deleteShareFileUrl, headers, ArrayList.class);
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    @Override
    public void deleteShareFile(String bucketName, String shareFileId) {
        this.deleteShareFile(bucketName, shareFileId, null);
    }

    @Override
    public void deleteShareFile(String shareFileId) {
        this.deleteShareFile(null, shareFileId, null);
    }

    /**
     * 获取我的分享
     *
     * @param bucketName
     * @param driveToken
     */
    @Override
    public List<SharedFile> getMyShareFiles(String bucketName, String driveToken) {
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        String deleteShareFileUrl = DmcUrl.getResourceUrl() + "/"
                + bucketName + "/findMyShareFiles";

        Map<String, String> headers = new HashMap<>();
        if (driveToken == null || "".equals(driveToken)) {
            String userToken = userManagerService.getUserToken(ServerSetting.getUser());
            headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        } else {
            headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
        }
        try {
            List<SharedFile> sharedFileList = new ArrayList<>();
            SharedFile[] sharedFiles = HttpRequestUtil.get(deleteShareFileUrl, headers, SharedFile[].class);
            if (sharedFiles.length > 0) {
                sharedFileList = Arrays.asList(sharedFiles);
            }
            return sharedFileList;
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    @Override
    public List<SharedFile> getMyShareFiles(String bucketName) {
        return this.getMyShareFiles(bucketName, null);
    }

    @Override
    public List<SharedFile> getMyShareFiles() {
        return this.getMyShareFiles(null, null);
    }

    @Override
    public IShareUploader uploadDocument(String bucketName, String localPath, FileInfo fileInfo, String driveToken) {
        if (localPath == null) {
            throw new IllegalArgumentException("The localPath is null ");
        }
        if (fileInfo == null) {
            throw new IllegalArgumentException("The fileInfo is null ");
        }
        return new ShareUploader(null, bucketName, localPath, fileInfo, driveToken);
    }

    @Override
    public IShareUploader uploadDocument(String bucketName, String localPath, FileInfo fileInfo) {
        return this.uploadDocument(bucketName, localPath, fileInfo, null);
    }

    @Override
    public IShareUploader uploadDocument(String localPath, FileInfo fileInfo) {
        return this.uploadDocument(null, localPath, fileInfo, null);
    }

    @Override
    public IShareUploader uploadDocument(String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken) {
        if (bytes == null) {
            throw new IllegalArgumentException("The bytes is null");
        }
        if (fileInfo == null) {
            throw new IllegalArgumentException("The fileInfo is null ");
        }
        return new ShareUploader(null, bucketName, bytes, fileInfo, driveToken);
    }

    @Override
    public IShareUploader uploadDocument(String bucketName, byte[] bytes, FileInfo fileInfo) {
        return this.uploadDocument(bucketName, bytes, fileInfo, null);
    }

    @Override
    public IShareUploader uploadDocument(byte[] bytes, FileInfo fileInfo) {
        return this.uploadDocument(null, bytes, fileInfo, null);
    }

}
