package com.digiwin.dmc.sdk.service.upload;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.NetworkOptions;
import com.digiwin.dmc.sdk.config.SDKConstants;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.FileInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.service.impl.UserManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.ObjectMapperUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @author chenzwd
 * @date : 2018-06-30 16:04
 * @Description:
 */
public class SegmentDocumentUploader implements ISegmentDocumentUploader {

    private static ExecutorService es = Executors.newCachedThreadPool();
    private static IUserManagerService userManagerService = UserManagerService.userInstance();
    private String tenantId;
    private String bucketName;
    private int bufferSize;
    private String driveToken;
    private UploadProcessState state;
    private FileInfo fileInfo;
    private FileInputStream fileInputStream;
    private InputStream inputStream;
    private IUpLoadCallbackable progressChangedCallback;
    private IUpLoadCallbackable completeCallback;
    private UploadProgressEventArgs eventArgs = new UploadProgressEventArgs();
    //分块传输或断点续传使用的上传会话Id
    private String uploadId;

    public SegmentDocumentUploader(String tenantId, String bucketName, String localPath, FileInfo fileInfo, String driveToken) {
        try {
            File file = new File(localPath);
            FileInputStream fileInputStream = new FileInputStream(file);
            this.fileInputStream = fileInputStream;
            eventArgs.setTotalBytes((int) file.length());
        } catch (Exception e) {
            throw new OperateException(e);
        }
        this.driveToken = driveToken;
        this.bucketName = bucketName;
        this.tenantId = tenantId;
        this.fileInfo = fileInfo;
        eventArgs.setPercentage(0);
        eventArgs.setCompletedBytes(0);
        this.state = UploadProcessState.NotStarted;
        this.bufferSize = NetworkOptions.Default.getPackageSize();
    }

    public SegmentDocumentUploader(String tenantId, String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken) {
        try {
            this.inputStream = new ByteArrayInputStream(bytes);
            eventArgs.setTotalBytes(inputStream.available());
        } catch (Exception e) {
            throw new OperateException(e);
        }
        this.driveToken = driveToken;
        this.bucketName = bucketName;
        this.tenantId = tenantId;
        this.fileInfo = fileInfo;
        eventArgs.setPercentage(0);
        eventArgs.setCompletedBytes(0);
        this.state = UploadProcessState.NotStarted;
        this.bufferSize = NetworkOptions.Default.getPackageSize();
    }


    /**
     * 异步上传Stream中的文件内容
     *
     * @return 上传文件Id
     */
    @Override
    public ISegmentDocumentUploader beginUpload() {

        if (state != UploadProcessState.NotStarted) {
            throw new OperateException("上传已开始，无法再次启动下载，请开启新的上传");
        }
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        String uploadResourceUrl = DmcUrl.getUploadResourceUrl()
                + bucketName + "/files/segment";
        try {
            Map<String, String> headers = new HashMap<>();
            if (driveToken == null || "".equals(driveToken)) {
                String userToken = userManagerService.getUserToken(ServerSetting.getUser());
                headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
            } else {
                headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
            }
            if (!StringUtil.isEmpty(tenantId)) {
                headers.put(SDKConstants.HTTP_HEADER_TENANTID_KEY, tenantId);
            }
            String json = ObjectMapperUtil.writeValueAsString(fileInfo);
            Map<String, String> result = HttpRequestUtil.postJson(uploadResourceUrl, json, headers, HashMap.class);
            if (result == null) {
                throw new Exception("Response is null");
            }
            this.uploadId = result.get("id");

            this.continueUpload();
            String finalBucketName = bucketName;
            Runnable runnable = () -> multipartUpload(finalBucketName);
            es.execute(runnable);

            return this;
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    public void uploadSegFile(String bucketName, int from, int count, byte[] buf) {
        if (StringUtil.isEmpty(bucketName)) {
            bucketName = ServerSetting.getBucketName();
        }
        String uploadResourceUrl = DmcUrl.getFilesResourceUrl() + bucketName +
                "/files/" + this.uploadId + "/" + from + "/" + (from + count - 1) + "/" + this.eventArgs.getTotalBytes();
        Map<String, String> headers = new HashMap<>();
//        if (driveToken==null||"".equals(driveToken)){
        String userToken = userManagerService.getUserToken(ServerSetting.getUser());
        headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
//        }else{
//            headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
//        }
        if (!StringUtil.isEmpty(tenantId)) {
            headers.put(SDKConstants.HTTP_HEADER_TENANTID_KEY, tenantId);
        }
        HttpRequestUtil.uploadSegFile(uploadResourceUrl, headers, count, buf);
    }


    public void multipartUpload(String bucketName) {
        byte[] buffer = new byte[bufferSize];
        int bytesReaded = 0;
        int totalBytes = eventArgs.getTotalBytes();

        while (bytesReaded < totalBytes) {
            try {
                bytesReaded = eventArgs.getCompletedBytes();
                if (state == UploadProcessState.Stopped) {
                    break;
                } else if (state == UploadProcessState.Paused) {
                    Thread.sleep(1000 * 60 * 60 * 24);
                    if (state == UploadProcessState.Paused) {
                        state = UploadProcessState.Timeout;
                        break;
                    }
                } else {
                    int currentReadLen = 0;
                    if (fileInputStream == null) {
                        currentReadLen = inputStream.read(buffer, 0, bufferSize);
                    } else {
                        currentReadLen = fileInputStream.read(buffer, 0, bufferSize);
                    }
                    if (currentReadLen == -1) {
                        break;
                    }
                    if (bytesReaded + bufferSize >= totalBytes) {
                        currentReadLen = totalBytes - bytesReaded;
                        this.uploadSegFile(bucketName, bytesReaded, currentReadLen, buffer);
                        eventArgs.setCompletedBytes(totalBytes);
                        eventArgs.setPercentage(1);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                        if (completeCallback != null) {
                            eventArgs.setFileId(uploadId);
                            completeCallback.callback(eventArgs);
                        }
                        return;
                    } else {
                        this.uploadSegFile(bucketName, bytesReaded, currentReadLen, buffer);
                        eventArgs.setCompletedBytes(bytesReaded + currentReadLen);
                        eventArgs.setPercentage(Math.round(eventArgs.getCompletedBytes() * 10000.0 / totalBytes) / 10000.0);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                    }
                }
            } catch (Exception e) {
                throw new OperateException(e);
            }
        }
    }

    /**
     * 设置响应进度
     *
     * @param callbackable
     */
    @Override
    public ISegmentDocumentUploader onProgressChanged(IUpLoadCallbackable callbackable) {
        this.progressChangedCallback = callbackable;
        return this;
    }

    /**
     * 设置完成
     *
     * @param callbackable
     */
    @Override
    public ISegmentDocumentUploader onCompleted(IUpLoadCallbackable callbackable) {
        this.completeCallback = callbackable;
        return this;
    }

    /**
     * 暂停上传
     */
    @Override
    public void pauseUpload() {
        state = UploadProcessState.Paused;
    }

    /**
     * 继续上传
     */
    @Override
    public void continueUpload() {
        state = UploadProcessState.Uploading;
    }

    /**
     * 停止上传
     */
    @Override
    public void stopUpload() {
        state = UploadProcessState.Stopped;
    }
}
