package com.digiwin.dap.middleware.dmc;

import com.digiwin.dap.middleware.dmc.model.*;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * Entry point interface of DAP Middleware DMC (Document Management Center)
 * <p>
 * Document Management Center (a.k.a DMC) is the massive, secure, low cost and highly
 * reliable public storage which could be accessed from anywhere at anytime via
 * REST APIs, SDKs. <br>
 * Developers could use DMC to create any services that need huge data storage
 * and access throughput, such as media sharing web apps, cloud storage service
 * or enterprise or personal data backup.
 * </p>
 *
 * @author fobgochod
 * @date 2021/4/23
 */
public interface DMC {

    /**
     * 复制一个文件到根目录
     *
     * @param fileId 被复制的文件ID
     * @return 新文件ID
     */
    String copy(String fileId) throws Exception;

    /**
     * 复制文件到一个指定目录
     *
     * @param fileId      被复制的文件ID
     * @param targetDirId 目标目录ID
     * @return 新文件ID
     */
    String copy(String fileId, String targetDirId) throws Exception;

    /**
     * 复制目录到根目录
     *
     * @param dirId 被复制的目录ID
     * @return 新目录ID
     */
    String copyDir(String dirId) throws Exception;

    /**
     * 复制目录到一个指定目录
     *
     * @param dirId       被复制的目录ID
     * @param targetDirId 目标目录ID
     * @return 新目录ID
     */
    String copyDir(String dirId, String targetDirId) throws Exception;

    /**
     * 批量复制
     *
     * @param fileIds     被复制的文件ID集合
     * @param dirIds      被复制的目录ID集合
     * @param targetDirId 目标目录ID
     */
    List<FileOpTree> batchCopy(List<String> fileIds, List<String> dirIds, String targetDirId) throws Exception;

    /**
     * 批量复制
     */
    List<FileOpTree> batchCopy(FileOpRequest request) throws Exception;

    /**
     * 剪切文件到根目录
     *
     * @param fileId 被剪切的文件ID
     */
    void move(String fileId) throws Exception;

    /**
     * 剪切文件到一个指定目录
     *
     * @param fileId      被剪切的文件ID
     * @param targetDirId 目标目录ID
     */
    void move(String fileId, String targetDirId) throws Exception;

    /**
     * 剪切目录到根目录
     *
     * @param dirId 被剪切的目录ID
     */
    void moveDir(String dirId) throws Exception;

    /**
     * 剪切目录到一个指定目录
     *
     * @param dirId       被剪切的目录ID
     * @param targetDirId 目标目录ID
     */
    void moveDir(String dirId, String targetDirId) throws Exception;

    /**
     * 批量剪切
     *
     * @param fileIds     被剪切的文件ID集合
     * @param dirIds      被剪切的目录ID集合
     * @param targetDirId 目标目录ID
     */
    void batchMove(List<String> fileIds, List<String> dirIds, String targetDirId) throws Exception;

    /**
     * 批量剪切
     */
    void batchMove(FileOpRequest request) throws Exception;

    /**
     * 永久删除指定文件
     *
     * @param fileId 被删除的文件ID
     */
    void delete(String fileId) throws Exception;

    /**
     * 永久删除指定文件
     *
     * @param dirId 被删除的目录ID
     */
    void deleteDir(String dirId) throws Exception;

    /**
     * 永久删除文件、目录
     *
     * @param fileIds 被删除的文件ID集合
     * @param dirIds  被删除的目录ID集合
     */
    void batchDelete(List<String> fileIds, List<String> dirIds) throws Exception;

    /**
     * 永久删除文件、目录
     */
    void batchDelete(BatchRequest request) throws Exception;

    /**
     * 普通上传
     *
     * @param file file
     */
    FileInfo upload(File file) throws Exception;

    /**
     * 普通上传
     *
     * @param file  文件
     * @param dirId 目录
     */
    FileInfo upload(File file, String dirId) throws Exception;

    /**
     * 普通上传，指定文件信息
     *
     * @param file     文件
     * @param fileInfo 文件信息
     */
    FileInfo upload(File file, FileInfo fileInfo) throws Exception;

    /**
     * 普通上传
     *
     * @param bytes    bytes 字节数组
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     */
    FileInfo upload(byte[] bytes, String filename) throws Exception;

    /**
     * 普通上传，指定文件信息
     *
     * @param bytes    字节数组
     * @param fileInfo 文件信息
     */
    FileInfo upload(byte[] bytes, FileInfo fileInfo) throws Exception;

    /**
     * 普通上传
     *
     * @param input    input 输入流
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     */
    FileInfo upload(InputStream input, String filename) throws Exception;

    /**
     * 普通上传，指定文件信息
     *
     * @param input    输入流
     * @param fileInfo 文件信息
     */
    FileInfo upload(InputStream input, FileInfo fileInfo) throws Exception;

    /**
     * 普通上传
     */
    FileInfo upload(UploadRequest request) throws Exception;

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param file   file
     */
    FileInfo coverUpload(String fileId, File file) throws Exception;

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param bytes  字节数组
     */
    FileInfo coverUpload(String fileId, byte[] bytes) throws Exception;

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param input  输入流
     */
    FileInfo coverUpload(String fileId, InputStream input) throws Exception;

    /**
     * 批量上传
     *
     * @param files 文件集合
     */
    List<FileInfo> batchUpload(List<File> files) throws Exception;

    /**
     * 批量上传
     * 文件、文件信息一一对应
     *
     * @param files     文件集合
     * @param fileInfos 文件信息集合
     */
    List<FileInfo> batchUpload(List<File> files, List<FileInfo> fileInfos) throws Exception;

    /**
     * 批量上传
     *
     * @param bytes     字节数组集合
     * @param filenames 文件名集合
     */
    List<FileInfo> batchUploadBytes(List<byte[]> bytes, List<String> filenames) throws Exception;

    /**
     * 批量上传
     *
     * @param bytes     字节数组集合
     * @param fileInfos 文件名集合
     */
    List<FileInfo> batchUploadBytes(List<byte[]> bytes, Map<String, FileInfo> fileInfos) throws Exception;

    /**
     * 批量上传
     *
     * @param inputs    输入流集合
     * @param filenames 文件名集合
     */
    List<FileInfo> batchUploadStream(List<InputStream> inputs, List<String> filenames) throws Exception;

    /**
     * 批量上传
     *
     * @param inputs    输入流集合
     * @param fileInfos 文件信息集合
     */
    List<FileInfo> batchUploadStream(List<InputStream> inputs, Map<String, FileInfo> fileInfos) throws Exception;

    /**
     * 批量上传
     */
    List<FileInfo> batchUpload(UploadMultiRequest request) throws Exception;

    /**
     * 分段上传
     *
     * @param file 文件
     * @return 文件ID(fileId)
     */
    String multipartUpload(File file) throws Exception;

    /**
     * 分段上传
     *
     * @param file     文件
     * @param partSize n*255KB 的整数倍
     * @return 文件ID(fileId)
     */
    String multipartUpload(File file, int partSize) throws Exception;

    /**
     * 分段上传
     *
     * @param file     文件
     * @param fileInfo 文件信息
     * @return 文件ID(fileId)
     */
    String multipartUpload(File file, FileInfo fileInfo) throws Exception;

    /**
     * 分段上传
     *
     * @param input    输入流
     * @param fileInfo 文件信息  fileName、size必填
     * @return 文件ID(fileId)
     */
    String multipartUpload(InputStream input, FileInfo fileInfo) throws Exception;

    /**
     * 分段上传
     *
     * @param file     文件
     * @param fileInfo 文件信息
     * @param partSize n*255KB 的整数倍
     * @return 文件ID(fileId)
     */
    String multipartUpload(File file, FileInfo fileInfo, int partSize) throws Exception;

    /**
     * 分段上传
     *
     * @param input    输入流
     * @param fileInfo 文件信息 fileName、size必填
     * @param partSize n*255KB 的整数倍
     * @return 文件ID(fileId)
     */
    String multipartUpload(InputStream input, FileInfo fileInfo, int partSize) throws Exception;

    /**
     * 分段上传
     */
    String multipartUpload(MultipartUploadRequest request) throws Exception;

    /**
     * 下载指定文件
     *
     * @param fileId 被下载的文件ID
     */
    byte[] download(String fileId) throws Exception;

    /**
     * 下载指定文件到流
     *
     * @param fileId 被下载的文件ID
     * @param output 输出流
     */
    void download(String fileId, OutputStream output) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param fileIds 被下载的文件ID集合
     * @param dirIds  被下载的目录ID集合
     */
    byte[] batchDownload(List<String> fileIds, List<String> dirIds) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param fileIds 被下载的文件ID集合
     * @param dirIds  被下载的目录ID集合
     * @param output  输出流
     */
    void batchDownload(List<String> fileIds, List<String> dirIds, OutputStream output) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     */
    byte[] batchDownload(BatchRequest request) throws Exception;

    /**
     * 根据图片ID获取缩放图片
     *
     * @param fileId 原图文件ID
     */
    List<ImageInfo> getImageInfo(String fileId) throws Exception;

    /**
     * 根据图片ID获取缩放图片
     *
     * @param fileId 原图文件ID
     * @param resize false-原图 true-缩放后的图片
     */
    List<ImageInfo> getImageInfo(String fileId, boolean resize) throws Exception;

    /**
     * 指定修改图片宽度，高度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     */
    ImageInfo resizeImageWidth(String fileId, int width) throws Exception;

    /**
     * 指定修改图片宽度，高度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @param share  是否分享
     */
    ImageInfo resizeImageWidth(String fileId, int width, boolean share) throws Exception;

    /**
     * 指定修改图片高度，宽度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param height 要设定的高度
     */
    ImageInfo resizeImageHeight(String fileId, int height) throws Exception;

    /**
     * 指定修改图片高度，宽度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param height 要设定的高度
     * @param share  是否分享
     */
    ImageInfo resizeImageHeight(String fileId, int height, boolean share) throws Exception;

    /**
     * 根据图片ID缩放图片
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @param height 要设定的高度
     */
    ImageInfo resizeImage(String fileId, int width, int height) throws Exception;

    /**
     * 根据图片ID缩放图片
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @param height 要设定的高度
     * @param share  是否分享
     */
    ImageInfo resizeImage(String fileId, int width, int height, boolean share) throws Exception;

    /**
     * 图片上传
     *
     * @param image image
     */
    ImageUpload resizeImage(File image, FileInfo imageInfo) throws Exception;

    /**
     * 图片上传 指定缩放宽度
     *
     * @param image     image
     * @param imageInfo 文件信息
     * @param width     要设定的宽度
     */
    ImageUpload resizeImageWidth(File image, FileInfo imageInfo, int width) throws Exception;

    /**
     * 图片上传 指定缩放高度
     *
     * @param image     image
     * @param imageInfo 文件信息
     * @param height    要设定的高度
     */
    ImageUpload resizeImageHeight(File image, FileInfo imageInfo, int height) throws Exception;

    /**
     * 图片上传并缩放
     *
     * @param image     image
     * @param imageInfo 文件信息
     * @param width     要设定的宽度
     * @param height    要设定的高度
     */
    ImageUpload resizeImage(File image, FileInfo imageInfo, int width, int height) throws Exception;

    /**
     * 删除文件到回收站
     *
     * @param fileId 被删除的文件ID
     */
    RecycleBin remove(String fileId) throws Exception;

    /**
     * 删除目录到回收站
     *
     * @param dirId 被删除的目录ID
     */
    RecycleBin removeDir(String dirId) throws Exception;

    /**
     * 批量删除文件、目录到回收站
     *
     * @param fileIds 被删除的文件ID集合
     * @param dirIds  被删除的目录ID集合
     */
    List<RecycleBin> batchRemove(List<String> fileIds, List<String> dirIds) throws Exception;

    /**
     * 批量删除文件、目录到回收站
     */
    List<RecycleBin> batchRemove(BatchRequest request) throws Exception;

    /**
     * 删除回收站指定文件
     *
     * @param recycleId 回收站ID
     */
    void deleteRecycleBin(String recycleId) throws Exception;

    /**
     * 批量删除回收站指定文件
     *
     * @param recycleIds 回收站ID集合
     */
    void batchDeleteRecycleBin(List<String> recycleIds) throws Exception;

    /**
     * 清空回收站
     */
    void clearRecycleBin() throws Exception;

    /**
     * 恢复回收站文件
     *
     * @param recycleId 回收站ID
     */
    void restoreRecycleBin(String recycleId) throws Exception;

    /**
     * 批量恢复回收站文件
     *
     * @param recycleIds 回收站ID集合
     */
    void batchRestoreRecycleBin(List<String> recycleIds) throws Exception;

    /**
     * 批量恢复回收站文件
     */
    void batchRestoreRecycleBin(BatchRequest request) throws Exception;

    /**
     * 分享指定文件
     *
     * @param fileId 被分享的文件ID
     */
    ShareInfo share(String fileId) throws Exception;

    /**
     * 删除分享记录
     *
     * @param shareId 分享ID
     */
    void deleteShare(String shareId) throws Exception;

    /**
     * 批量删除分享记录
     *
     * @param shareIds 分享ID集合
     */
    void batchDeleteShare(List<String> shareIds) throws Exception;


    /**********************************************************************
     *                                                                    *
     *                             目录基本信息操作                          *
     *                                                                    *
     **********************************************************************/

    /**
     * 创建目录
     *
     * @param dirInfo 目录信息
     */
    DirInfo createDir(DirInfo dirInfo) throws Exception;

    /**
     * 修改目录信息
     *
     * @param dirInfo 目录信息
     */
    DirInfo modDirInfo(DirInfo dirInfo) throws Exception;

    /**
     * 获取目录信息
     *
     * @param dirId 目录ID
     */
    DirInfo getDirInfo(String dirId) throws Exception;

    /**
     * 获取目录信息
     *
     * @param parentDirId 父目录ID
     * @param dirName     目录名称
     */
    DirInfo getDirInfoByName(String parentDirId, String dirName) throws Exception;


    /**********************************************************************
     *                                                                    *
     *                             文件基本信息操作                          *
     *                                                                    *
     **********************************************************************/

    /**
     * 修改文件信息
     *
     * @param fileInfo 文件信息
     */
    FileInfo modFileInfo(FileInfo fileInfo) throws Exception;

    /**
     * 设置文件过期时间
     *
     * @param fileId     文件ID
     * @param expireDate 过期时间
     */
    FileInfo modFileExpireDate(String fileId, LocalDateTime expireDate) throws Exception;

    /**
     * 修改文件目录
     *
     * @param fileId 文件ID
     * @param dirId  目录ID
     */
    FileInfo modFileDirectoryId(String fileId, String dirId) throws Exception;

    /**
     * 查询文件信息
     *
     * @param fileId 文件ID
     */
    FileInfo getFileInfo(String fileId) throws Exception;

    /**
     * 查询目录下的文件信息
     *
     * @param dirId 目录ID
     */
    List<FileInfo> getFileInfos(String dirId) throws Exception;

    /**
     * 查询指定IDs的文件信息
     *
     * @param fileIds 文件ID集合
     */
    List<FileInfo> getFileInfos(List<String> fileIds) throws Exception;

    /**
     * 获取userToken
     */
    String getToken();

    /**
     * 获取指定租户对应的userToken
     */
    String getToken(String tenantId);

    /**
     * 切换登录租户到默认(default)租户
     * <p>
     * 无租户即default租户，可以操作所有租户文件
     */
    void switchTenantId();

    /**
     * Switches to another tenantId
     * <p>
     * 切换登录租户，当前client生效
     * <ol>
     *     <li>上传文件均归属该租户</li>
     *     <li>没有权限操作其他租户的文件</li>
     *     <li>和{@link #setTenantId(String)}不要混用，调用后会使{@link #setTenantId(String)}失效</li>
     * </ol>
     *
     * @param tenantId the tenantId to switch to。
     */
    void switchTenantId(String tenantId);

    /**
     * 设置上传租户，当前线程生效
     *
     * <ol>
     *     <li>只有默认(default)租户才能使用</li>
     *     <li>文件都上传到该租户</li>
     *     <li>本质是更新fileInfo一个属性tenantId</li>
     *     <li>和{@link #switchTenantId()}不要混用，切换租户，该设定会被清空</li>
     * </ol>
     *
     * @param tenantId 租户ID
     */
    DMC setTenantId(String tenantId);

    /**
     * Shuts down the DMC instance (release all resources) The DMC instance is
     * not usable after its shutdown() is called.
     */
    void shutdown();
}
