package com.digiwin.dap.middleware.dmc;

import com.digiwin.dap.middleware.dmc.internal.DMCConfig;

/**
 * Fluent builder for DMC Client. Use of the builder is preferred over using
 * constructors of the client class.
 *
 * @author fobgochod
 * @date 2021/4/23
 */
public interface DMCBuilder {

    /**
     * create a new {@link DMCClient} instance.
     *
     * @return {@link DMCClient} instance
     */
    static DMCBuilder create() {
        return new DMCClientBuilder(false);
    }

    /**
     * create a new {@link DMCClient} instance.
     * <p>
     * 通过 {@link #build(DMCConfig)}、{@link #build(DMCConfig, ClientConfiguration)} 创建实例，该方法lazy参数不生效<br />
     * 应该通过 {@link DMCConfig#setLazy(boolean)} 直接设定
     *
     * @param lazy 延时创建token
     * @return {@link DMCClient} instance
     */
    static DMCBuilder create(boolean lazy) {
        return new DMCClientBuilder(lazy);
    }

    /**
     * create a new {@link DMCClient} instance.
     * <p>
     * 兼容：使用旧版配置信息 {@link com.digiwin.dmc.sdk.config.ServerSetting}
     *
     * @return {@link DMCClient} instance
     */
    @Deprecated
    DMC build();

    /**
     * Uses the specified tenantId to create a new {@link DMCClient} instance.
     * <p>
     * 兼容：使用旧版配置信息 {@link com.digiwin.dmc.sdk.config.ServerSetting}
     *
     * @param tenantId IAM tenantId
     * @return {@link DMCClient} instance
     */
    @Deprecated
    DMC build(String tenantId);

    /**
     * Uses the specified tenantId to flush the specified client instance.
     * <p>
     * 可以使用 client.switchTenantId(tenantId)切换
     *
     * @param tenantId IAM tenantId
     * @return {@link DMCClient} instance
     */
    @Deprecated
    DMC build(DMC client, String tenantId);

    /**
     * Uses the parameters to create a new {@link DMCClient} instance.
     *
     * @param uri      DMC server uri
     * @param username DMC username
     * @param password DMC user password
     * @param bucket   DMC database
     * @return {@link DMCClient} instance
     */
    DMC build(String uri, String username, String password, String bucket);

    /**
     * Uses the parameters to create a new {@link DMCClient} instance.
     *
     * @param uri       DMC server uri
     * @param username  DMC username
     * @param password  DMC user password
     * @param bucket    DMC database
     * @param appToken  IAM sys appToken
     * @param appSecret IAM sys appSecret
     * @return {@link DMCClient} instance
     */
    DMC build(String uri, String username, String password, String bucket, String appToken, String appSecret);

    /**
     * Uses the specified tenantId to create a new {@link DMCClient} instance.
     *
     * @param uri      DMC server uri
     * @param username DMC username
     * @param password DMC user password
     * @param tenantId IAM tenantId
     * @param bucket   DMC database
     * @return {@link DMCClient} instance
     */
    DMC build(String uri, String username, String password, String tenantId, String bucket);

    /**
     * Uses the specified tenantId to create a new {@link DMCClient} instance.
     *
     * @param uri       DMC server uri
     * @param username  DMC username
     * @param password  DMC user password
     * @param tenantId  IAM tenantId
     * @param bucket    DMC database
     * @param appToken  IAM sys appToken
     * @param appSecret IAM sys appSecret
     * @return {@link DMCClient} instance
     */
    DMC build(String uri, String username, String password, String tenantId, String bucket, String appToken, String appSecret);

    /**
     * Uses the DMCConfig to create a new {@link DMCClient} instance.
     *
     * @param config 基础配置信息
     * @return {@link DMCClient} instance
     */
    DMC build(DMCConfig config);

    /**
     * Uses the DMCConfig to create a new {@link DMCClient} instance.
     *
     * @param config       基础配置信息
     * @param clientConfig Client configurations for accessing to DMC services
     * @return {@link DMCClient} instance
     */
    DMC build(DMCConfig config, ClientConfiguration clientConfig);
}
