package com.digiwin.dap.middleware.dmc.common.comm;

import com.digiwin.dap.middleware.dmc.ClientConfiguration;
import com.digiwin.dap.middleware.dmc.ClientException;
import com.digiwin.dap.middleware.dmc.ServiceException;
import com.digiwin.dap.middleware.dmc.common.auth.AuthCache;
import com.digiwin.dap.middleware.dmc.common.utils.CodingUtils;
import com.digiwin.dap.middleware.dmc.internal.DMCConfig;
import com.digiwin.dap.middleware.dmc.internal.DMCUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.List;

/**
 * Abstract service client that provides interfaces to access DMC services.
 *
 * @author fobgochod
 * @date 2021/4/25
 */
public abstract class ServiceClient {

    protected static final Logger logger = LoggerFactory.getLogger(ServiceClient.class);
    protected ClientConfiguration clientConfig;

    protected ServiceClient(ClientConfiguration clientConfig) {
        this.clientConfig = clientConfig;
    }

    public ClientConfiguration getClientConfiguration() {
        return this.clientConfig;
    }

    protected abstract ResponseMessage sendRequestCore(RequestMessage request, ExecutionContext context) throws IOException;

    public abstract DMCConfig getConfig();

    public abstract AuthCache getAuthCache();

    public abstract void setAuthCache(String tenantId);

    public abstract void shutdown();

    /**
     * Send HTTP request with specified context to fobgochod and wait for HTTP
     * response.
     */
    public ResponseMessage sendRequest(RequestMessage request, ExecutionContext context)
            throws ServiceException, ClientException {

        CodingUtils.assertParameterNotNull(request, "request");
        CodingUtils.assertParameterNotNull(context, "context");

        try {
            return sendRequestImpl(request, context);
        } finally {
            // Close the request stream as well after the request is completed.
            try {
                request.close();
            } catch (IOException ex) {
                logger.error("Unexpected io exception when trying to close http request: ", ex);
                throw new ClientException("Unexpected io exception when trying to close http request: ", ex);
            }
        }
    }

    public ResponseMessage sendRequestImpl(RequestMessage request, ExecutionContext context) throws ServiceException, ClientException {
        ResponseMessage response = null;
        try {
            // Step 1. Preprocess HTTP request.
            handleRequest(request, context.getRequestHandlers());

            // Step 2. Send HTTP request to DMC.
            response = sendRequestCore(request, context);

            // Step 3. Preprocess HTTP response.
            handleResponse(response, context.getResponseHandlers());

            return response;
        } catch (ServiceException sex) {
            logger.error("[Server]Unable to execute HTTP request: ", sex);
            // Notice that the response should not be closed in the
            // finally block because if the request is successful,
            // the response should be returned to the callers.
            closeResponseSilently(response);
            throw sex;
        } catch (ClientException cex) {
            logger.error("[Client]Unable to execute HTTP request: ", cex);
            closeResponseSilently(response);
            throw cex;
        } catch (Exception ex) {
            logger.error("[Unknown]Unable to execute HTTP request: ", ex);
            closeResponseSilently(response);
            throw new ClientException(DMCUtils.COMMON_RESOURCE_MANAGER.getFormattedString("ConnectionError", ex.getMessage()), ex);
        }
    }

    private void handleRequest(RequestMessage message, List<RequestHandler> requestHandlers)
            throws ServiceException, ClientException {
        for (RequestHandler h : requestHandlers) {
            h.handle(message);
        }
    }

    private void handleResponse(ResponseMessage response, List<ResponseHandler> responseHandlers)
            throws ServiceException, ClientException {
        for (ResponseHandler h : responseHandlers) {
            h.handle(response);
        }
    }

    private void closeResponseSilently(ResponseMessage response) {
        if (response != null) {
            try {
                response.close();
            } catch (IOException ioe) {
                /* silently close the response. */
            }
        }
    }
}


