package com.digiwin.dap.middleware.dmc.common.utils;

import com.digiwin.dap.middleware.dmc.ClientErrorCode;
import com.digiwin.dap.middleware.dmc.ClientException;
import com.digiwin.dap.middleware.dmc.DMCErrorCode;
import com.digiwin.dap.middleware.dmc.DMCException;
import com.digiwin.dap.middleware.dmc.internal.DMCUtils;
import com.digiwin.dap.middleware.dmc.internal.model.StdError;
import org.apache.http.NoHttpResponseException;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.NonRepeatableRequestException;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.conn.HttpHostConnectException;

import java.io.IOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

/**
 * A simple factory used for creating instances of <code>ClientException</code>
 * and <code>fobgochodException</code>.
 *
 * @author fobgochod
 * @date 2021/4/23
 */
public class ExceptionFactory {

    public static ClientException createNetworkException(IOException ex) {
        String requestId = "Unknown";
        String errorCode = ClientErrorCode.UNKNOWN;

        if (ex instanceof SocketTimeoutException) {
            errorCode = ClientErrorCode.SOCKET_TIMEOUT;
        } else if (ex instanceof SocketException) {
            errorCode = ClientErrorCode.SOCKET_EXCEPTION;
        } else if (ex instanceof ConnectTimeoutException) {
            errorCode = ClientErrorCode.CONNECTION_TIMEOUT;
        } else if (ex instanceof UnknownHostException) {
            errorCode = ClientErrorCode.UNKNOWN_HOST;
        } else if (ex instanceof HttpHostConnectException) {
            errorCode = ClientErrorCode.CONNECTION_REFUSED;
        } else if (ex instanceof NoHttpResponseException) {
            errorCode = ClientErrorCode.CONNECTION_TIMEOUT;
        } else if (ex instanceof ClientProtocolException) {
            Throwable cause = ex.getCause();
            if (cause instanceof NonRepeatableRequestException) {
                errorCode = ClientErrorCode.NONREPEATABLE_REQUEST;
                return new ClientException(cause.getMessage(), errorCode, requestId, cause);
            }
        }
        return new ClientException(ex.getMessage(), errorCode, requestId, ex);
    }

    public static DMCException createInvalidResponseException(String requestId, String rawResponseError,
                                                              Throwable cause) {
        return createInvalidResponseException(requestId,
                DMCUtils.COMMON_RESOURCE_MANAGER.getFormattedString("FailedToParseResponse", cause.getMessage()),
                rawResponseError);
    }

    public static DMCException createInvalidResponseException(String requestId, String message,
                                                              String rawResponseError) {
        return createDMCException(requestId, DMCErrorCode.INVALID_RESPONSE, message, rawResponseError);
    }

    public static DMCException createDMCException(String requestId, String errorCode, String message,
                                                  String rawResponseError) {
        return new DMCException(message, errorCode, requestId, null, null, null, null, rawResponseError);
    }

    public static DMCException createDMCException(String requestId, String errorCode, String message) {
        return new DMCException(message, errorCode, requestId, null, null, null, null);
    }

    public static DMCException createUnknownDMCException(String requestId, int statusCode) {
        String message = "No body in response, http status code " + statusCode;
        return new DMCException(message, ClientErrorCode.UNKNOWN, requestId, null, null, null, null);
    }

    public static DMCException createDMCException(StdError errorResult) {
        return new DMCException(String.format("%s：%s", errorResult.getErrorMessage(), errorResult.getMessage()),
                errorResult.getErrorCode());
    }
}
