package com.digiwin.dap.middleware.dmc.internal;

import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.util.AESCipher;

/**
 * Defines a builder that adds a config to the DMCConfig entity.
 *
 * @author fobgochod
 * @date 2023/3/9 16:18
 */
public final class DMCConfigBuilder {

    private String uri = ServerSetting.getServiceUrl();
    private String bucket = ServerSetting.getBucketName();
    private String username = ServerSetting.getIdentityName();
    private String password = ServerSetting.getIdentityPwd();
    private String tenantId = DMCConstants.DEFAULT_TENANT;
    private boolean lazy;
    private String appToken = ServerSetting.getAppToken();
    private String appSecret = ServerSetting.getAppSecret();

    /**
     * Enforce the use of factory methods.
     */
    private DMCConfigBuilder() {
    }

    public static DMCConfigBuilder create() {
        return new DMCConfigBuilder();
    }

    public static DMCConfigBuilder create(String uri, String username, String password, String bucket) {
        return create().uri(uri).username(username).password(password).bucket(bucket);
    }

    public DMCConfigBuilder uri(String uri) {
        this.uri = uri;
        return this;
    }

    public DMCConfigBuilder bucket(String bucket) {
        this.bucket = bucket;
        return this;
    }

    public DMCConfigBuilder username(String username) {
        this.username = username;
        return this;
    }

    public DMCConfigBuilder password(String password) {
        this.password = password;
        return this;
    }

    public DMCConfigBuilder tenantId(String tenantId) {
        this.tenantId = tenantId;
        return this;
    }

    public DMCConfigBuilder lazy(boolean lazy) {
        this.lazy = lazy;
        return this;
    }

    public DMCConfigBuilder appToken(String appToken) {
        this.appToken = appToken;
        return this;
    }

    public DMCConfigBuilder appSecret(String appSecret) {
        this.appSecret = appSecret;
        return this;
    }

    public DMCConfig build() {
        DMCConfig config = new DMCConfig();
        config.setUri(uri);
        config.setBucket(bucket);
        config.setUsername(username);
        config.setPassword(password);
        if (password != null) {
            config.setPwdhash(AESCipher.getSHA256StrJava(password));
        }
        config.setTenantId(tenantId);
        config.setLazy(lazy);
        config.setAppToken(appToken);
        config.setAppSecret(appSecret);
        return config;
    }
}
