package com.digiwin.dap.middleware.dmc.internal;

import com.digiwin.dap.middleware.dmc.common.comm.ResponseMessage;
import com.digiwin.dap.middleware.dmc.common.parser.ResponseParseException;
import com.digiwin.dap.middleware.dmc.common.parser.ResponseParser;
import com.digiwin.dap.middleware.dmc.common.utils.IOUtils;
import com.digiwin.dap.middleware.dmc.internal.model.StdData;
import com.digiwin.dap.middleware.dmc.model.*;
import com.digiwin.dmc.sdk.util.JsonUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.util.List;

/**
 * A collection of parsers that parse HTTP responses into corresponding human-readable results.
 *
 * @author fobgochod
 * @date 2021/4/22
 */
public final class ResponseParsers {

    public static final EmptyResponseParser emptyResponseParser = new EmptyResponseParser();
    public static final LoginResponseParser loginResponseParser = new LoginResponseParser();
    public static final UploadResponseParser uploadResponseParser = new UploadResponseParser();
    public static final UploadMultiResponseParser uploadMultiResponseParser = new UploadMultiResponseParser();
    public static final UploadMultipartResponseParser uploadMultipartResponseParser = new UploadMultipartResponseParser();
    public static final DownloadResponseParser downloadResponseParser = new DownloadResponseParser();
    public static final DeleteResponseParser deleteResponseParser = new DeleteResponseParser();
    public static final DeleteMultiResponseParser deleteMultiResponseParser = new DeleteMultiResponseParser();
    public static final CopyResponseParser copyResponseParser = new CopyResponseParser();
    public static final CopyDirResponseParser copyDirResponseParser = new CopyDirResponseParser();
    public static final CopyMultiResponseParser copyMultiResponseParser = new CopyMultiResponseParser();
    public static final ShareMultiResponseParser shareMultiResponseParser = new ShareMultiResponseParser();
    public static final ImageInfoResponseParser imageInfoResponseParser = new ImageInfoResponseParser();
    public static final ImageShrinkResponseParser imageShrinkResponseParser = new ImageShrinkResponseParser();
    public static final ImageUploadResponseParser imageUploadResponseParser = new ImageUploadResponseParser();
    public static final DirInfoResponseParser dirInfoResponseParser = new DirInfoResponseParser();
    public static final FileInfoResponseParser fileInfoResponseParser = new FileInfoResponseParser();
    public static final FileInfosResponseParser fileInfosResponseParser = new FileInfosResponseParser();

    private static final ObjectMapper objectMapper = JsonUtils.createObjectMapper();

    public static final class EmptyResponseParser implements ResponseParser<ResponseMessage> {
        @Override
        public ResponseMessage parse(ResponseMessage response) throws ResponseParseException {
            // Close response and return it directly without parsing.
            DMCUtils.safeCloseResponse(response);
            return response;
        }
    }

    public static final class LoginResponseParser implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                LoginToken result = objectMapper.readValue(response.getContent(), LoginToken.class);
                return result.getUserToken();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class UploadResponseParser implements ResponseParser<FileInfo> {
        @Override
        public FileInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<FileInfo> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<FileInfo>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class UploadMultiResponseParser implements ResponseParser<List<FileInfo>> {
        @Override
        public List<FileInfo> parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<List<FileInfo>> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<List<FileInfo>>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class UploadMultipartResponseParser implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                FileInfo fileInfo = objectMapper.readValue(response.getContent(), FileInfo.class);
                return fileInfo.getId();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class DownloadResponseParser implements ResponseParser<byte[]> {
        @Override
        public byte[] parse(ResponseMessage response) throws ResponseParseException {
            try {
                return IOUtils.readStreamAsByteArray(response.getContent());
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class DeleteResponseParser implements ResponseParser<RecycleBin> {
        @Override
        public RecycleBin parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<RecycleBin> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<RecycleBin>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class DeleteMultiResponseParser implements ResponseParser<List<RecycleBin>> {
        @Override
        public List<RecycleBin> parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<List<RecycleBin>> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<List<RecycleBin>>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class CopyResponseParser implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<FileOpId> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<FileOpId>>() {
                });
                return stdData.getData().getFileId();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class CopyDirResponseParser implements ResponseParser<String> {
        @Override
        public String parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<FileOpId> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<FileOpId>>() {
                });
                return stdData.getData().getDirId();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class CopyMultiResponseParser implements ResponseParser<List<FileOpTree>> {
        @Override
        public List<FileOpTree> parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<List<FileOpTree>> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<List<FileOpTree>>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class ShareMultiResponseParser implements ResponseParser<ShareInfo> {
        @Override
        public ShareInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<ShareInfo> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<ShareInfo>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class ImageInfoResponseParser implements ResponseParser<List<ImageInfo>> {
        @Override
        public List<ImageInfo> parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<List<ImageInfo>> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<List<ImageInfo>>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class ImageShrinkResponseParser implements ResponseParser<ImageInfo> {
        @Override
        public ImageInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<ImageInfo> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<ImageInfo>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class ImageUploadResponseParser implements ResponseParser<ImageUpload> {
        @Override
        public ImageUpload parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<ImageUpload> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<ImageUpload>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class DirInfoResponseParser implements ResponseParser<DirInfo> {
        @Override
        public DirInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<DirInfo> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<DirInfo>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class FileInfoResponseParser implements ResponseParser<FileInfo> {
        @Override
        public FileInfo parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<FileInfo> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<FileInfo>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }

    public static final class FileInfosResponseParser implements ResponseParser<List<FileInfo>> {
        @Override
        public List<FileInfo> parse(ResponseMessage response) throws ResponseParseException {
            try {
                StdData<List<FileInfo>> stdData = objectMapper.readValue(response.getContent(), new TypeReference<StdData<List<FileInfo>>>() {
                });
                return stdData.getData();
            } catch (Exception e) {
                throw new ResponseParseException(e.getMessage(), e);
            } finally {
                DMCUtils.safeCloseResponse(response);
            }
        }
    }
}
