package com.digiwin.dmc.sdk.service.impl;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.DriveAuthItem;
import com.digiwin.dmc.sdk.entity.DriveAuthRequest;
import com.digiwin.dmc.sdk.entity.DriveAuthRequestAction;
import com.digiwin.dmc.sdk.entity.DriveAuthRequestObjectType;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IDriveTokenService;
import com.digiwin.dmc.sdk.util.*;

import java.util.HashMap;
import java.util.Map;

/**
 * @author chenxsa
 */
public class DriveTokenServiceImpl implements IDriveTokenService {

    private static final String EMPTY_OBJECT_ID_STR = "00000000-0000-0000-0000-000000000000";
    private final Map<Integer, String> cache = new HashMap<>();

    /**
     * 单一实例
     */
    public static IDriveTokenService instance() {
        return Holder.instance;
    }

    /**
     * 获取DriveToken
     *
     * @param authRequest 需要操作的文件或文件夹
     * @return token
     */
    @Override
    public String getDriveToken(DriveAuthRequest authRequest) {
        if (authRequest == null) {
            throw new IllegalArgumentException("参数：authRequest is null");
        }
        String driveToken;
        int hashcode = authRequest.hashCode();
        if (cache.containsKey(hashcode)) {
            driveToken = cache.get(hashcode);
            if (!JwtUtil.isExpires(driveToken)) {
                return driveToken;
            }
        }

        String tokenUrl = DmcUrl.getTokenResourceUrl();
        try {
            Map<String, String> headers = HttpUtils.setHeader();
            String json = ObjectMapperUtil.writeValueAsString(authRequest);
            HashMap responseEntity = HttpRequestUtil.postJson(tokenUrl, json, headers, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("driveToken")) {
                cache.remove(hashcode);
                return null;
            } else {
                driveToken = responseEntity.get("driveToken").toString();
                cache.put(hashcode, driveToken);
                return driveToken;
            }
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 获取DriveToken
     *
     * @param authRequest 需要操作的文件或文件夹
     * @param tenantId    租户Id
     * @return DriveToken
     */
    @Deprecated
    @Override
    public String getDriveTokenByTenant(String tenantId, DriveAuthRequest authRequest) {
        if (authRequest == null) {
            throw new IllegalArgumentException("参数：authRequest is null");
        }
        String driveToken;
        int hashcode = authRequest.hashCode();
        if (cache.containsKey(hashcode)) {
            driveToken = cache.get(hashcode);
            if (!JwtUtil.isExpires(driveToken)) {
                return driveToken;
            }
        }

        String tokenUrl = DmcUrl.getTokenResourceUrl();
        try {
            Map<String, String> headers = HttpUtils.setHeader(null, tenantId);
            String json = ObjectMapperUtil.writeValueAsString(authRequest);
            HashMap responseEntity = HttpRequestUtil.postJson(tokenUrl, json, headers, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("driveToken")) {
                cache.remove(hashcode);
                return null;
            } else {
                driveToken = responseEntity.get("driveToken").toString();
                cache.put(hashcode, driveToken);
                return driveToken;
            }
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 获取目录读DriveToken
     *
     * @param dirId 目录id
     * @return token
     */
    @Override
    public String getDirWriteDriveToken(String dirId) {
        if (StringUtil.isEmpty(dirId)) {
            dirId = EMPTY_OBJECT_ID_STR;
        }
        return getDriveToken(dirId,
                DriveAuthRequestAction.Write,
                DriveAuthRequestObjectType.Directory);
    }

    /**
     * 获取目录读DriveToken
     *
     * @param dirId 目录id
     * @return token
     */
    @Override
    public String getDirReadDriveToken(String dirId) {
        if (StringUtil.isEmpty(dirId)) {
            dirId = EMPTY_OBJECT_ID_STR;
        }
        return getDriveToken(dirId,
                DriveAuthRequestAction.Read,
                DriveAuthRequestObjectType.Directory);
    }

    /**
     * 获取文件读DriveToken
     *
     * @param fileId 文件id
     * @return token
     */
    @Override
    public String getFileReadDriveToken(String fileId) {
        if (StringUtil.isEmpty(fileId)) {
            throw new IllegalArgumentException("fileId is null");
        }
        return getDriveToken(fileId,
                DriveAuthRequestAction.Read,
                DriveAuthRequestObjectType.File);
    }

    String getDriveToken(String id, DriveAuthRequestAction driveAuthRequestAction,
                         DriveAuthRequestObjectType driveAuthRequestObjectType) {
        //首字母大写
        DriveAuthRequest driveAuthRequest = new DriveAuthRequest();
        DriveAuthItem driveAuthItem = new DriveAuthItem();
        driveAuthItem.setBucket(ServerSetting.getBucketName());
        driveAuthItem.setActionName(driveAuthRequestAction);
        driveAuthItem.setObjectType(driveAuthRequestObjectType);
        driveAuthItem.setObjectId(id);
        driveAuthRequest.getItems().add(driveAuthItem);
        return getDriveToken(driveAuthRequest);
    }

    /**
     * 获取文件写DriveToken
     *
     * @param fileId 文件id
     * @return token
     */
    @Override
    public String getFileWriteDriveToken(String fileId) {
        if (StringUtil.isEmpty(fileId)) {
            throw new IllegalArgumentException("fileId is null");
        }
        return getDriveToken(fileId,
                DriveAuthRequestAction.Write,
                DriveAuthRequestObjectType.File);
    }

    private static class Holder { // lazy class
        static final IDriveTokenService instance = new DriveTokenServiceImpl();
    }
}
