package com.digiwin.dmc.sdk.service.impl;

import com.digiwin.dap.middleware.dmc.common.auth.TokenCache;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.UserInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.identity.UserIdentity;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.HttpUtils;
import com.digiwin.dmc.sdk.util.ObjectMapperUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class UserManagerService implements IUserManagerService {

    /**
     * 单一实例
     */
    public static IUserManagerService userInstance() {
        return Holder.instance;
    }

    /**
     * 创建用户
     *
     * @param userInfo 用户信息
     * @return String
     */
    @Override
    public String createUser(UserInfo userInfo) {
        if (StringUtil.checkSpecialId(userInfo.getName())) {
            throw new IllegalArgumentException("userName为空或包含特殊字符.号！");
        }
        Map<String, String> userObj = new HashMap<>();
        userObj.put("name", userInfo.getName());
        userObj.put("pwdHash", userInfo.getPassword());
        userObj.put("displayName", userInfo.getDisplayName());
        userObj.put("builtinRole", userInfo.getRole().toString());
        try {
            String uri = String.format("%s/api/dmc/v1/users", ServerSetting.getServiceUrl());

            Map<String, String> headers = HttpUtils.setHeaderAdmin();
            Map responseEntity = HttpRequestUtil.postJson(uri, ObjectMapperUtil.writeValueAsString(userObj), headers, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("user-id")) {
                return "";
            }
            return responseEntity.get("user-id").toString();
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 查询用户
     *
     * @param userName 用户名
     * @return 类
     */
    @Override
    public UserInfo getUser(String userName) {
        if (userName == null || userName.length() == 0) {
            throw new IllegalArgumentException("参数userName is null");
        }
        String uri = String.format("%s/api/dmc/v1/users/%s", ServerSetting.getServiceUrl(), userName);
        Map<String, String> headers = HttpUtils.setHeaderAdmin();
        return HttpRequestUtil.get(uri, headers, UserInfo.class);
    }

    /**
     * 查询
     *
     * @return list
     */
    @Override
    public List<UserInfo> getUsers() {
        String uri = String.format("%s/api/dmc/v1/users", ServerSetting.getServiceUrl());
        Map<String, String> headers = HttpUtils.setHeaderAdmin();
        List users = HttpRequestUtil.get(uri, headers, ArrayList.class);
        List<UserInfo> userInfos = new ArrayList<>();
        if (users != null) {
            for (Object user : users) {
                try {
                    String json = ObjectMapperUtil.writeValueAsString(user);
                    UserInfo userInfo = ObjectMapperUtil.readValue(json, UserInfo.class);
                    userInfos.add(userInfo);
                } catch (IOException e) {
                    throw new OperateException(e);
                }
            }
        }
        return userInfos;
    }

    /**
     * 判断用户是否存在
     *
     * @param userName 用户名
     * @return boolean
     */
    @Override
    public String isExistUser(String userName) {
        if (userName == null) {
            throw new IllegalArgumentException("参数userIdentity is null");
        }
        String uri = String.format("%s/api/dmc/v1/users/%s/isExist", ServerSetting.getServiceUrl(), userName);
        Map<String, String> headers = HttpUtils.setHeaderAdmin();
        Map isExist = HttpRequestUtil.get(uri, headers, HashMap.class);
        if (isExist == null || !isExist.containsKey("isExist")) {
            return null;
        }
        return isExist.get("isExist").toString();
    }

    /**
     * 获取用户token
     *
     * @param userIdentity 用户身份
     * @return String
     */
    @Override
    public String getUserToken(UserIdentity userIdentity) {
        if (userIdentity == null) {
            throw new IllegalArgumentException("参数userIdentity is null");
        }
        final String uniqueKey = userIdentity.uniqueKey();
        String userToken = TokenCache.get(uniqueKey);
        if (userToken != null) {
            return userToken;
        }
        TokenCache.remove(uniqueKey);

        Map<String, String> parameters = new HashMap<>(3);
        parameters.put("tenantId", userIdentity.getTenantId());
        parameters.put("username", userIdentity.getUserName());
        parameters.put("pwdhash", userIdentity.getPasswordHash());
        try {
            String uri = String.format("%s/api/dmc/v1/auth/login", ServerSetting.getServiceUrl());

            String body = ObjectMapperUtil.writeValueAsString(parameters);
            Map<String, String> headers = HttpUtils.setHeaderApp();
            Map<String, String> response = HttpRequestUtil.postJson(uri, body, headers, HashMap.class);

            TokenCache.put(uniqueKey, response.get("userToken"));
            return TokenCache.get(uniqueKey);
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    private static class Holder { // lazy class
        static final IUserManagerService instance = new UserManagerService();
    }
}
