package com.digiwin.dmc.sdk.service.upload;

import com.digiwin.dmc.sdk.config.NetworkOptions;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.FileInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.util.*;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;

/**
 * 分段上传
 * step1: /api/dmc/v1/buckets/{bucket}/files/segment
 * step2: /api/dmc/v1/buckets/{bucket}/files/{fileId}/{from}/{to}/{total}
 *
 * @author chenzwd
 * @date : 2018-06-30 16:04
 */
public class SegmentDocumentUploader implements ISegmentDocumentUploader {

    private String discardTenantId;
    private String bucketName;
    private FileInfo fileInfo;
    private String driveToken;
    private int bufferSize = NetworkOptions.Default.getPackageSize();
    private InputStream inputStream;
    private IUpLoadCallbackable completeCallback;
    private IUpLoadCallbackable progressChangedCallback;
    private UploadProcessState state = UploadProcessState.NotStarted;
    private UploadProgressEventArgs eventArgs = new UploadProgressEventArgs();
    /**
     * 分块传输或断点续传使用的上传会话Id
     */
    private String uploadId;

    private SegmentDocumentUploader(String discardTenantId, String bucketName, FileInfo fileInfo, String driveToken) {
        this.discardTenantId = discardTenantId;
        this.bucketName = bucketName;
        this.fileInfo = fileInfo;
        this.driveToken = driveToken;
    }

    public SegmentDocumentUploader(String discardTenantId, String bucketName, String localPath, FileInfo fileInfo, String driveToken) {
        this(discardTenantId, bucketName, fileInfo, driveToken);
        try {
            this.inputStream = Files.newInputStream(Paths.get(localPath));
            eventArgs.setTotalBytes(inputStream.available());
        } catch (IOException e) {
            throw new OperateException(e);
        }
        eventArgs.setPercentage(0);
        eventArgs.setCompletedBytes(0);
    }

    public SegmentDocumentUploader(String discardTenantId, String bucketName, byte[] bytes, FileInfo fileInfo, String driveToken) {
        this(discardTenantId, bucketName, fileInfo, driveToken);
        try {
            this.inputStream = new ByteArrayInputStream(bytes);
            eventArgs.setTotalBytes(inputStream.available());
        } catch (IOException e) {
            throw new OperateException(e);
        }
        eventArgs.setPercentage(0);
        eventArgs.setCompletedBytes(0);
    }


    /**
     * 异步上传Stream中的文件内容
     *
     * @return 上传文件Id
     */
    @Override
    public ISegmentDocumentUploader beginUpload() {
        if (state != UploadProcessState.NotStarted) {
            throw new OperateException("上传已开始，无法再次启动下载，请开启新的上传");
        }
        try {
            bucketName = ArgumentUtils.getBucketName(bucketName);
            Map<String, String> headers = HttpUtils.setHeader(driveToken, discardTenantId);

            this.uploadId = createEmptyFile(bucketName, headers);

            this.continueUpload();

            Runnable runnable = () -> {
                multipartUpload(bucketName, headers);
            };
            ExecutorUtil.newExecutor().execute(runnable);
            return this;
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    private String createEmptyFile(String bucketName, Map<String, String> headers) throws Exception {
        String uri = String.format("%s/api/dmc/v1/buckets/%s/files/segment", ServerSetting.getServiceUrl(), bucketName);
        String json = ObjectMapperUtil.writeValueAsString(fileInfo);
        Map<String, String> result = HttpRequestUtil.postJson(uri, json, headers, HashMap.class);
        if (result == null) {
            throw new Exception("Response is null");
        }
        return result.get("id");

    }

    public void uploadSegFile(String bucketName, Map<String, String> headers, int from, int count, byte[] buf) {
        String uri = String.format("%s/api/dmc/v1/buckets/%s/files/%s/%s/%s/%s", ServerSetting.getServiceUrl(), bucketName, this.uploadId, from, (from + count - 1), this.eventArgs.getTotalBytes());

        HttpRequestUtil.uploadSegFile(uri, headers, count, buf);
    }


    public void multipartUpload(String bucketName, Map<String, String> headers) {
        byte[] buffer = new byte[bufferSize];
        int bytesReaded = 0;
        int totalBytes = eventArgs.getTotalBytes();

        while (bytesReaded < totalBytes) {
            try {
                bytesReaded = eventArgs.getCompletedBytes();
                if (state == UploadProcessState.Stopped) {
                    break;
                } else if (state == UploadProcessState.Paused) {
                    Thread.sleep(1000 * 60 * 60 * 24);
                    if (state == UploadProcessState.Paused) {
                        state = UploadProcessState.Timeout;
                        break;
                    }
                } else {
                    int currentReadLen = inputStream.read(buffer, 0, bufferSize);
                    if (currentReadLen == -1) {
                        break;
                    }
                    if (bytesReaded + bufferSize >= totalBytes) {
                        currentReadLen = totalBytes - bytesReaded;
                        this.uploadSegFile(bucketName, headers, bytesReaded, currentReadLen, buffer);
                        eventArgs.setCompletedBytes(totalBytes);
                        eventArgs.setPercentage(1);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                        if (completeCallback != null) {
                            eventArgs.setFileId(uploadId);
                            completeCallback.callback(eventArgs);
                        }
                        return;
                    } else {
                        this.uploadSegFile(bucketName, headers, bytesReaded, currentReadLen, buffer);
                        eventArgs.setCompletedBytes(bytesReaded + currentReadLen);
                        eventArgs.setPercentage(Math.round(eventArgs.getCompletedBytes() * 10000.0 / totalBytes) / 10000.0);
                        if (progressChangedCallback != null) {
                            progressChangedCallback.callback(eventArgs);
                        }
                    }
                }
            } catch (Exception e) {
                throw new OperateException(e);
            }
        }
    }

    /**
     * 设置响应进度
     */
    @Override
    public ISegmentDocumentUploader onProgressChanged(IUpLoadCallbackable callbackable) {
        this.progressChangedCallback = callbackable;
        return this;
    }

    /**
     * 设置完成
     */
    @Override
    public ISegmentDocumentUploader onCompleted(IUpLoadCallbackable callbackable) {
        this.completeCallback = callbackable;
        return this;
    }

    /**
     * 暂停上传
     */
    @Override
    public void pauseUpload() {
        state = UploadProcessState.Paused;
    }

    /**
     * 继续上传
     */
    @Override
    public void continueUpload() {
        state = UploadProcessState.Uploading;
    }

    /**
     * 停止上传
     */
    @Override
    public void stopUpload() {
        state = UploadProcessState.Stopped;
    }
}
