package com.digiwin.dmc.sdk.util;

import com.digiwin.dmc.sdk.serializer.*;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;

/**
 * Json序列化
 *
 * @author fobgochod
 * @date 2021/4/20
 */
public final class JsonUtils {

    private static final Logger logger = LoggerFactory.getLogger(JsonUtils.class);
    private static final String JSON_SERIALIZER = "【序列化】：{}";
    private static final String JSON_DESERIALIZER = "【反序列化】：{}";
    private static volatile ObjectMapper objectMapper = null;

    static {
        createObjectMapper();
    }


    public static String writeValueAsString(Object data) {
        try {
            return objectMapper.writeValueAsString(data);
        } catch (JsonProcessingException e) {
            logger.error(JSON_SERIALIZER, e.getMessage());
            return null;
        }
    }

    public static <T> T readValue(String content, Class<T> valueType) {
        try {
            return objectMapper.readValue(content, valueType);
        } catch (IOException e) {
            logger.error(JSON_DESERIALIZER, e.getMessage());
            return null;
        }
    }

    public static <T> T readValue(String json, TypeReference<T> valueTypeRef) {
        try {
            return objectMapper.readValue(json, valueTypeRef);
        } catch (IOException e) {
            logger.error(JSON_DESERIALIZER, e.getMessage());
            return null;
        }
    }

    public static <T> T readValue(InputStream src, Class<T> valueType) {
        try {
            return createObjectMapper().readValue(src, valueType);
        } catch (IOException e) {
            logger.error(JSON_DESERIALIZER, e.getMessage());
            return null;
        }
    }

    public static <T> T readValue(InputStream src, TypeReference<T> valueTypeRef) {
        try {
            return objectMapper.readValue(src, valueTypeRef);
        } catch (IOException e) {
            logger.error(JSON_DESERIALIZER, e.getMessage());
            return null;
        }
    }

    public static ObjectMapper createObjectMapper() {
        if (objectMapper == null) {
            synchronized (JsonUtils.class) {
                if (objectMapper == null) {
                    JavaTimeModule javaTimeModule = new JavaTimeModule();
                    javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer());
                    javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
                    javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer());
                    javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
                    javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer());
                    javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
                    javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer());
                    javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());
                    objectMapper = new ObjectMapper();
                    objectMapper.registerModule(javaTimeModule);
                    objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
                    objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
                    objectMapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_CONTROL_CHARS, true);
                    objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
                }
            }
        }
        return objectMapper;
    }
}
