package com.digiwin.dap.middleware.dmc;

import com.digiwin.dap.middleware.commons.core.codec.Base64;
import com.digiwin.dap.middleware.dmc.common.security.RSAAESDecryptionDownloadResponse;
import com.digiwin.dap.middleware.dmc.common.security.RSAAESDigitalEnvelopeBuilder;
import com.digiwin.dap.middleware.dmc.common.security.encryption.strategy.EncryptionStrategyFactory;
import com.digiwin.dap.middleware.dmc.common.utils.FileCopyUtils;
import com.digiwin.dap.middleware.dmc.common.utils.FileUtils;
import com.digiwin.dap.middleware.dmc.common.utils.StreamUtils;
import com.digiwin.dap.middleware.dmc.internal.DMCConfig;
import com.digiwin.dap.middleware.dmc.internal.DMCConstants;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadPartV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.model.UploadV1Request;
import com.digiwin.dap.middleware.dmc.internal.client.operation.DownloadV1EncryptionOperation;
import com.digiwin.dap.middleware.dmc.internal.client.operation.UploadV1EncryptionOperation;
import com.digiwin.dap.middleware.dmc.internal.operation.DMCBucketOperation;
import com.digiwin.dap.middleware.dmc.internal.operation.DMCDownloadEncryptionOperation;
import com.digiwin.dap.middleware.dmc.internal.operation.DMCUploadEncryptionOperation;
import com.digiwin.dap.middleware.dmc.model.*;
import com.digiwin.dap.middleware.commons.crypto.AES;
import com.digiwin.dap.middleware.commons.crypto.RSA;
import com.digiwin.dap.middleware.commons.crypto.constant.KeySizeEnum;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.KeyPair;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;

/**
 * DMCEncryptionClient is used to do client-side encryption.
 *
 * @author michael
 * @since 5.3.0.0
 */
public class DMCEncryptionClient extends DMCClient {

    private DMCBucketOperation dmcBucketOperation;
    private DMCUploadEncryptionOperation dmcUploadEncryptionOperation;
    private DMCDownloadEncryptionOperation dmcDownloadEncryptionOperation;
    private UploadV1EncryptionOperation uploadV1EncryptionOperation;
    private DownloadV1EncryptionOperation downloadV1EncryptionOperation;

    /**
     * 兼容老版本,客户端加密不兼容老版本,使用其他构造器
     *
     * @param config
     */
    DMCEncryptionClient(DMCConfig config) {
        super(config);
    }

    /**
     * 兼容老版本,客户端加密不兼容老版本,使用其他构造器
     *
     * @param config
     */
    DMCEncryptionClient(DMCConfig config, ClientConfiguration clientConfig) {
        super(config, clientConfig);
    }

    public void initOperations() {
        super.initOperations();
        byte[] aesKey = AES.generateKey(KeySizeEnum.AES_256.getValue());
        this.dmcBucketOperation = new DMCBucketOperation(this.serviceClient);
        this.dmcUploadEncryptionOperation = new DMCUploadEncryptionOperation(this.serviceClient,
                new EncryptionStrategyFactory(aesKey),
                new RSAAESDigitalEnvelopeBuilder(aesKey));
        KeyPair keyPair = RSA.generateKeyPair(KeySizeEnum.RSA_1024.getValue());
        String publicKey = Base64.encode(keyPair.getPublic().getEncoded());
        String privateKey = Base64.encode(keyPair.getPrivate().getEncoded());
        this.dmcDownloadEncryptionOperation = new DMCDownloadEncryptionOperation(this.serviceClient, publicKey,
                new RSAAESDecryptionDownloadResponse(privateKey));

        this.uploadV1EncryptionOperation = new UploadV1EncryptionOperation(serviceClient,
                new EncryptionStrategyFactory(aesKey),
                new RSAAESDigitalEnvelopeBuilder(aesKey)
        );
        this.downloadV1EncryptionOperation = new DownloadV1EncryptionOperation(serviceClient, publicKey,
                new RSAAESDecryptionDownloadResponse(privateKey));
    }

    @Override
    public FileInfo upload(File file) throws Exception {
        return this.upload(file, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public FileInfo upload(File file, String dirId) throws Exception {
        return this.upload(file, null, dirId);
    }

    @Override
    public FileInfo upload(File file, String filename, String dirId) throws Exception {
        return this.upload(file, new FileInfo(filename, dirId));
    }

    @Override
    public FileInfo upload(File file, FileInfo fileInfo) throws Exception {
        UploadRequest request = new UploadRequest();
        request.setFile(file);
        request.setFileInfo(fileInfo);
        return this.upload(request);
    }

    @Override
    public FileInfo upload(byte[] bytes, String filename) throws Exception {
        return this.upload(bytes, filename, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public FileInfo upload(byte[] bytes, String filename, String dirId) throws Exception {
        return this.upload(bytes, new FileInfo(filename, dirId));
    }

    @Override
    public FileInfo upload(byte[] bytes, FileInfo fileInfo) throws Exception {
        UploadRequest request = new UploadRequest();
        request.setBytes(bytes);
        request.setFileInfo(fileInfo);
        return this.upload(request);
    }

    @Override
    public FileInfo upload(InputStream input, String filename) throws Exception {
        return this.upload(input, filename, DMCConstants.EMPTY_UUID_STR);
    }

    @Override
    public FileInfo upload(InputStream input, String filename, String dirId) throws Exception {
        return this.upload(input, new FileInfo(filename, dirId));
    }

    @Override
    public FileInfo upload(InputStream input, FileInfo fileInfo) throws Exception {
        UploadRequest request = new UploadRequest();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        return this.upload(request);
    }

    @Override
    public FileInfo upload(UploadRequest request) throws Exception {
        Bucket bucket = dmcBucketOperation.getBucketInfoByName();
        request.setPublicKey(bucket.getPublicKey());
        return dmcUploadEncryptionOperation.upload(request);
    }

    @Override
    public FileInfo coverUpload(String fileId, File file) throws Exception {
        UploadCoverRequest request = new UploadCoverRequest();
        request.setFileId(fileId);
        request.setFile(file);
        Bucket bucket = dmcBucketOperation.getBucketInfoByName();
        request.setPublicKey(bucket.getPublicKey());
        return dmcUploadEncryptionOperation.coverUpload(request);
    }

    @Override
    public FileInfo coverUpload(String fileId, byte[] bytes) throws Exception {
        FileInfo fileInfo = this.getFileInfo(fileId);
        if (fileInfo == null) {
            throw new DMCException(String.format("文件[%s]不存在", fileId));
        }
        UploadCoverRequest request = new UploadCoverRequest();
        request.setFileId(fileId);
        request.setFileInfo(fileInfo);
        request.setBytes(bytes);
        Bucket bucket = dmcBucketOperation.getBucketInfoByName();
        request.setPublicKey(bucket.getPublicKey());
        return dmcUploadEncryptionOperation.coverUpload(request);
    }

    @Override
    public FileInfo coverUpload(String fileId, InputStream input) throws Exception {
        FileInfo fileInfo = this.getFileInfo(fileId);
        if (fileInfo == null) {
            throw new DMCException(String.format("文件[%s]不存在", fileId));
        }
        UploadCoverRequest request = new UploadCoverRequest();
        request.setFileId(fileId);
        request.setFileInfo(fileInfo);
        request.setInput(input);
        Bucket bucket = dmcBucketOperation.getBucketInfoByName();
        request.setPublicKey(bucket.getPublicKey());
        return dmcUploadEncryptionOperation.coverUpload(request);
    }

    @Override
    public List<FileInfo> batchUpload(List<File> files) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setFiles(files);
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUpload(List<File> files, List<FileInfo> fileInfos) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setFiles(files);
        request.setFileInfos(fileInfos);
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadBytes(List<byte[]> bytes, List<String> filenames) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setBytes(bytes);
        request.setFileInfos(toFileInfo(filenames));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadBytes(List<byte[]> bytes, Map<String, FileInfo> fileInfos) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setBytes(bytes);
        request.setFileInfos(toFileInfo(fileInfos));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadStream(List<InputStream> inputs, List<String> filenames) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setInputs(inputs);
        request.setFileInfos(toFileInfo(filenames));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUploadStream(List<InputStream> inputs, Map<String, FileInfo> fileInfos) throws Exception {
        UploadMultiRequest request = new UploadMultiRequest();
        request.setInputs(inputs);
        request.setFileInfos(toFileInfo(fileInfos));
        return this.batchUpload(request);
    }

    @Override
    public List<FileInfo> batchUpload(UploadMultiRequest request) throws Exception {
        Bucket bucket = dmcBucketOperation.getBucketInfoByName();
        request.setPublicKey(bucket.getPublicKey());
        return dmcUploadEncryptionOperation.uploadMulti(request);
    }

    @Override
    public String multipartUpload(File file) throws Exception {
        return multipartUpload(file, DMCConstants.DEFAULT_PART_SIZE);
    }

    @Override
    public String multipartUpload(File file, int partSize) throws Exception {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setFileName(file.getName());
        return multipartUpload(file, fileInfo, partSize);
    }

    @Override
    public String multipartUpload(File file, FileInfo fileInfo) throws Exception {
        return multipartUpload(file, fileInfo, DMCConstants.DEFAULT_PART_SIZE);
    }

    @Override
    public String multipartUpload(File file, FileInfo fileInfo, int partSize) throws Exception {
        MultipartUploadRequest request = new MultipartUploadRequest();
        request.setFile(file);
        request.setFileInfo(fileInfo);
        request.setFileLength(file.length());
        request.setPartSize(partSize);
        return multipartUpload(request);
    }

    @Override
    public String multipartUpload(InputStream input, FileInfo fileInfo) throws Exception {
        return multipartUpload(input, fileInfo, DMCConstants.DEFAULT_PART_SIZE);
    }

    @Override
    public String multipartUpload(InputStream input, FileInfo fileInfo, int partSize) throws Exception {
        MultipartUploadRequest request = new MultipartUploadRequest();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        request.setFileLength(fileInfo.getSize());
        request.setPartSize(partSize);
        return multipartUpload(request);
    }

    @Override
    public String multipartUpload(MultipartUploadRequest request) throws Exception {
        Bucket bucket = dmcBucketOperation.getBucketInfoByName();
        request.setPublicKey(bucket.getPublicKey());
        return dmcUploadEncryptionOperation.multipartUpload(request);
    }

    @Override
    public byte[] download(String fileId) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        return dmcDownloadEncryptionOperation.download(request);
    }

    @Override
    public void download(String fileId, String filePath) throws Exception {
        String fileName = this.getFileInfo(fileId).getFileName();
        this.download(fileId, filePath, fileName);
    }

    @Override
    public void download(String fileId, String filePath, String fileName) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        byte[] bytes = dmcDownloadEncryptionOperation.download(request);
        FileCopyUtils.copy(bytes, FileUtils.newOutputStream(filePath, fileName));
    }

    @Override
    public void download(final String fileId, final OutputStream output) throws Exception {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        byte[] bytes = dmcDownloadEncryptionOperation.download(request);
        output.write(bytes);
    }

    @Override
    public byte[] batchDownload(List<String> fileIds, List<String> dirIds) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        return this.batchDownload(request);
    }

    @Override
    public void batchDownload(List<String> fileIds, List<String> dirIds, String filePath) throws Exception {
        String fileName = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")) + ".zip";
        this.batchDownload(fileIds, dirIds, filePath, fileName);
    }

    @Override
    public void batchDownload(List<String> fileIds, List<String> dirIds, String filePath, String fileName) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        byte[] bytes = this.batchDownload(request);
        FileCopyUtils.copy(bytes, FileUtils.newOutputStream(filePath, fileName));
    }

    @Override
    public void batchDownload(List<String> fileIds, List<String> dirIds, OutputStream output) throws Exception {
        BatchRequest request = new BatchRequest();
        request.setFileIds(fileIds);
        request.setDirIds(dirIds);
        byte[] bytes = this.batchDownload(request);
        output.write(bytes);
    }

    @Override
    public byte[] batchDownload(BatchRequest request) throws Exception {
        return dmcDownloadEncryptionOperation.downloadMulti(request);
    }

    @Override
    public FileInfo uploadV1(InputStream input, FileInfo fileInfo) {
        UploadV1Request request = new UploadV1Request();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        return uploadV1(request);
    }

    @Override
    public FileInfo uploadV1(UploadV1Request request) {
        try {
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1EncryptionOperation.upload(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public FileInfo uploadCoverV1(String fileId, InputStream input) {
        UploadV1Request request = new UploadV1Request();
        request.setInput(input);
        request.setFileId(fileId);
        return uploadCoverV1(request);
    }

    @Override
    public FileInfo uploadCoverV1(UploadV1Request request) {
        try {
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1EncryptionOperation.uploadCover(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public ShareInfo uploadShareV1(InputStream input, FileInfo fileInfo) {
        UploadV1Request request = new UploadV1Request();
        request.setInput(input);
        request.setFileInfo(fileInfo);
        return uploadShareV1(request);
    }

    @Override
    public ShareInfo uploadShareV1(UploadV1Request request) {
        try {
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1EncryptionOperation.uploadShare(request);
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String claimFileIdV1(UploadPartV1Request request) {
        try {
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1EncryptionOperation.claimFileId(request).getId();
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String uploadPartV1(UploadPartV1Request request) {
        try {
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1EncryptionOperation.uploadPart(request).getId();
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public String uploadPartCoverV1(UploadPartV1Request request) {
        try {
            Bucket bucket = dmcBucketOperation.getBucketInfoByName();
            request.setPublicKey(bucket.getPublicKey());
            return uploadV1EncryptionOperation.uploadPartCover(request).getId();
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public void downloadV1(String fileId, OutputStream output) {
        DownloadRequest request = new DownloadRequest();
        request.setFileId(fileId);
        request.setOutput(output);
        downloadV1(request);
    }

    @Override
    public byte[] downloadV1(DownloadRequest request) {
        try {
            byte[] bytes = downloadV1EncryptionOperation.download(request);
            if (request.getOutput() != null) {
                StreamUtils.copy(bytes, request.getOutput());
            }
            return bytes;
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }

    @Override
    public byte[] downloadPartV1(DownloadPartRequest request) {
        try {
            byte[] bytes = downloadV1EncryptionOperation.downloadPart(request);
            StreamUtils.copy(bytes, request.getOutput());
            return bytes;
        } catch (Exception e) {
            throw new DMCException(e);
        }
    }
}
