package com.digiwin.dap.middleware.dmc.event;

import com.digiwin.dap.middleware.dmc.internal.model.WebServiceRequest;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

public abstract class ProgressInputStream extends FilterInputStream {

    public static InputStream inputStreamForRequest(InputStream is, WebServiceRequest req) {
        return req == null ? is : new RequestProgressInputStream(is, req.getProgressListener());
    }

    public static InputStream inputStreamForResponse(InputStream is, WebServiceRequest req) {
        return req == null ? is : new ResponseProgressInputStream(is, req.getProgressListener());
    }

    private static final int DEFAULT_NOTIFICATION_THRESHOLD = 8 * 1024;

    private final ProgressListener listener;
    private final int notifyThreshold;
    private int unNotifiedByteCount;
    private boolean hasBeenRead;
    private boolean doneEOF;
    private long notifiedByteCount;

    public ProgressInputStream(InputStream is, ProgressListener listener) {
        this(is, listener, DEFAULT_NOTIFICATION_THRESHOLD);
    }

    public ProgressInputStream(InputStream is, ProgressListener listener, int notifyThreshold) {
        super(is);
        if (is == null || listener == null) {
            throw new IllegalArgumentException();
        }
        this.listener = listener;
        this.notifyThreshold = notifyThreshold;
    }

    protected void onFirstRead() {
    }

    protected void onEOF() {
    }

    protected void onClose() {
        eof();
    }

    protected void onReset() {
    }

    protected void onNotifyBytesRead() {
    }

    private void onBytesRead(int bytesRead) {
        unNotifiedByteCount += bytesRead;
        if (unNotifiedByteCount >= notifyThreshold) {
            onNotifyBytesRead();
            notifiedByteCount += unNotifiedByteCount;
            unNotifiedByteCount = 0;
        }
    }

    @Override
    public int read() throws IOException {
        if (!hasBeenRead) {
            onFirstRead();
            hasBeenRead = true;
        }
        int ch = super.read();
        if (ch == -1)
            eof();
        else
            onBytesRead(1);
        return ch;
    }

    @Override
    public void reset() throws IOException {
        super.reset();
        onReset();
        unNotifiedByteCount = 0;
        notifiedByteCount = 0;
    }

    @Override
    public int read(byte b[]) throws IOException {
        return read(b, 0, b.length);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        if (!hasBeenRead) {
            onFirstRead();
            hasBeenRead = true;
        }
        int bytesRead = super.read(b, off, len);
        if (bytesRead == -1)
            eof();
        else
            onBytesRead(bytesRead);
        return bytesRead;
    }

    private void eof() {
        if (doneEOF)
            return;
        onEOF();
        unNotifiedByteCount = 0;
        doneEOF = true;
    }

    public final InputStream getWrappedInputStream() {
        return in;
    }

    protected final int getUnNotifiedByteCount() {
        return unNotifiedByteCount;
    }

    protected final long getNotifiedByteCount() {
        return notifiedByteCount;
    }

    @Override
    public void close() throws IOException {
        onClose();
        super.close();
    }

    public final ProgressListener getListener() {
        return listener;
    }
}
