package com.digiwin.dap.middleware.dmc.common.comm;

import com.digiwin.dap.middleware.dmc.HttpMethod;
import com.digiwin.dap.middleware.dmc.internal.DMCConstants;
import org.apache.http.HttpRequest;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.Args;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.ConnectException;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 重试策略
 *
 * @author fobgochod
 * @date 2023/7/31 11:08
 * @see org.apache.http.impl.client.DefaultHttpRequestRetryHandler
 * @see org.apache.http.impl.client.StandardHttpRequestRetryHandler
 */
public class CustomHttpRequestRetryHandler implements HttpRequestRetryHandler {

    public static final CustomHttpRequestRetryHandler INSTANCE = new CustomHttpRequestRetryHandler();
    public static final String RETRY_ATTRIBUTE = DMCConstants.DMC_SDK + "retry";

    private static final Logger logger = LoggerFactory.getLogger(CustomHttpRequestRetryHandler.class);
    private static final long RETRY_INTERVAL_TIME = 100;
    /**
     * <a href="https://www.mscharhag.com/api-design/http-idempotent-safe">HTTP methods: Idempotency and Safety</a>
     */
    private static final Map<String, Boolean> idempotentMethods;
    private static final Set<Class<? extends IOException>> nonRetriableClasses;

    static {
        idempotentMethods = new ConcurrentHashMap<>();
        idempotentMethods.put(HttpMethod.GET.name(), Boolean.TRUE);
        idempotentMethods.put(HttpMethod.HEAD.name(), Boolean.TRUE);
        idempotentMethods.put(HttpMethod.PUT.name(), Boolean.TRUE);
        idempotentMethods.put(HttpMethod.DELETE.name(), Boolean.TRUE);
        idempotentMethods.put(HttpMethod.OPTIONS.name(), Boolean.TRUE);
        idempotentMethods.put(HttpMethod.TRACE.name(), Boolean.TRUE);

        nonRetriableClasses = new HashSet<>();
        nonRetriableClasses.add(InterruptedIOException.class);
        nonRetriableClasses.add(UnknownHostException.class);
        nonRetriableClasses.add(ConnectException.class);
        nonRetriableClasses.add(SSLException.class);
    }

    /**
     * the number of times a method will be retried
     */
    private final int retryCount;
    /**
     * Whether or not methods that have successfully sent their request will be retried
     */
    private final boolean requestSentRetryEnabled;

    /**
     * Default constructor
     */
    public CustomHttpRequestRetryHandler(final int retryCount, final boolean requestSentRetryEnabled) {
        this.retryCount = retryCount;
        this.requestSentRetryEnabled = requestSentRetryEnabled;
    }

    /**
     * Default constructor
     */
    public CustomHttpRequestRetryHandler() {
        this(3, false);
    }

    @Override
    public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
        Args.notNull(exception, "Exception parameter");
        Args.notNull(context, "HTTP context");
        if (executionCount > this.retryCount) {
            // Do not retry if over max retry count
            return false;
        }
        if (nonRetriableClasses.contains(exception.getClass())) {
            return false;
        }
        for (final Class<? extends IOException> rejectException : nonRetriableClasses) {
            if (rejectException.isInstance(exception)) {
                return false;
            }
        }
        final HttpClientContext clientContext = HttpClientContext.adapt(context);
        final HttpRequest request = clientContext.getRequest();

        if (handleAsIdempotent(request)) {
            // Retry if the request is considered idempotent
            return trueDelayReturn(exception, request, executionCount);
        } else {
            if (exception instanceof CustomStatusRetryException
                    && Boolean.TRUE.equals(context.getAttribute(RETRY_ATTRIBUTE))) {
                return trueDelayReturn(exception, request, executionCount);
            }
        }

        if (!clientContext.isRequestSent() || this.requestSentRetryEnabled) {
            // Retry if the request has not been sent fully or
            // if it's OK to retry methods that have been sent
            return trueDelayReturn(exception, request, executionCount);
        }
        // otherwise do not retry
        return false;
    }

    /**
     * 是否幂等
     *
     * @param request 请求信息
     * @return true or false
     */
    protected boolean handleAsIdempotent(final HttpRequest request) {
        final String method = request.getRequestLine().getMethod().toUpperCase(Locale.ROOT);
        return idempotentMethods.containsKey(method);
    }

    /**
     * 延迟重试
     *
     * @param exception      异常
     * @param request        请求信息
     * @param executionCount 次数
     * @return true
     */
    protected boolean trueDelayReturn(IOException exception, HttpRequest request, int executionCount) {
        try {
            long sleep = executionCount * RETRY_INTERVAL_TIME;
            logger.warn("Retry request, execution count: {}, sleep: {}ms, uri: {}, exception: {}", executionCount, sleep, request.getRequestLine(), exception.getMessage());
            TimeUnit.MILLISECONDS.sleep(sleep);
        } catch (InterruptedException ignored) {
        }
        return true;
    }
}
