package com.digiwin.dap.middleware.dmc.internal;

import com.digiwin.dap.middleware.dmc.HttpMethod;
import com.digiwin.dap.middleware.dmc.common.comm.RequestMessage;
import com.digiwin.dap.middleware.dmc.internal.model.WebServiceRequest;
import org.apache.http.HttpEntity;

import java.io.InputStream;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * HTTP request message builder.
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public class DMCRequestMessageBuilder {

    private URI endpoint;
    private HttpMethod method = HttpMethod.GET;
    private Map<String, String> headers = new HashMap<>();
    private HttpEntity entity;
    private boolean whitelist;
    private boolean forceRetry;
    private InputStream input;
    private long inputSize = 0;
    private WebServiceRequest originalRequest;
    private String digitalEnvelope;
    private String clientPublicKey;
    private boolean dmcSecurity = Boolean.FALSE;
    private boolean onlySysAccountAccess;

    public static DMCRequestMessageBuilder create() {
        return new DMCRequestMessageBuilder();
    }

    public URI getEndpoint() {
        return endpoint;
    }

    public DMCRequestMessageBuilder setEndpoint(URI endpoint) {
        this.endpoint = endpoint;
        return this;
    }

    public HttpMethod getMethod() {
        return method;
    }

    public DMCRequestMessageBuilder setMethod(HttpMethod method) {
        this.method = method;
        return this;
    }

    public Map<String, String> getHeaders() {
        return Collections.unmodifiableMap(headers);
    }

    public DMCRequestMessageBuilder setHeaders(Map<String, String> headers) {
        this.headers = headers;
        return this;
    }

    public DMCRequestMessageBuilder addHeader(String key, String value) {
        headers.put(key, value);
        return this;
    }

    public DMCRequestMessageBuilder setEntity(HttpEntity entity) {
        this.entity = entity;
        return this;
    }

    public DMCRequestMessageBuilder setWhitelist(boolean whitelist) {
        this.whitelist = whitelist;
        return this;
    }

    public DMCRequestMessageBuilder setForceRetry(boolean forceRetry) {
        this.forceRetry = forceRetry;
        return this;
    }

    public InputStream getInput() {
        return input;
    }

    public DMCRequestMessageBuilder setInput(InputStream instream) {
        this.input = instream;
        return this;
    }

    public DMCRequestMessageBuilder setInputSize(long inputSize) {
        this.inputSize = inputSize;
        return this;
    }

    public DMCRequestMessageBuilder setOriginalRequest(WebServiceRequest originalRequest) {
        this.originalRequest = originalRequest;
        return this;
    }

    public DMCRequestMessageBuilder setDigitalEnvelope(String digitalEnvelope) {
        this.digitalEnvelope = digitalEnvelope;
        return this;
    }

    public DMCRequestMessageBuilder setDMCSecurity(Boolean dmcSecurity) {
        this.dmcSecurity = dmcSecurity;
        return this;
    }

    public DMCRequestMessageBuilder setClientPublicKey(String publicKey) {
        this.clientPublicKey = publicKey;
        return this;
    }

    public DMCRequestMessageBuilder setOnlySysAccountAccess(boolean onlySysAccountAccess) {
        this.onlySysAccountAccess = onlySysAccountAccess;
        return this;
    }

    public RequestMessage build() {
        Map<String, String> sentHeaders = new HashMap<>(this.headers);
        sentHeaders.put(DMCHeaders.CONTENT_LENGTH, this.inputSize + "");
        sentHeaders.put(DMCHeaders.DMC_SECURITY, String.valueOf(this.dmcSecurity));
        sentHeaders.put(DMCHeaders.DIGITAL_ENVELOPE, this.digitalEnvelope);
        sentHeaders.put(DMCHeaders.PUBLIC_KEY, this.clientPublicKey);

        RequestMessage request = new RequestMessage(this.originalRequest);
        request.setEndpoint(this.endpoint);
        request.setMethod(this.method);
        request.setHeaders(sentHeaders);
        request.setContent(this.input);
        request.setContentLength(this.inputSize);
        request.setHttpEntity(this.entity);
        request.setWhitelist(this.whitelist);
        request.setForceRetry(this.forceRetry);
        request.setOnlySysAccountAccess(this.onlySysAccountAccess);
        return request;
    }
}
