package com.digiwin.dap.middleware.dmc.internal.operation;

import com.digiwin.dap.middleware.dmc.DMCException;
import com.digiwin.dap.middleware.dmc.HttpMethod;
import com.digiwin.dap.middleware.dmc.common.comm.RequestMessage;
import com.digiwin.dap.middleware.dmc.common.comm.ServiceClient;
import com.digiwin.dap.middleware.dmc.common.security.DigitalEnvelopeBuilder;
import com.digiwin.dap.middleware.dmc.common.security.encryption.strategy.EncryptionStrategyFactory;
import com.digiwin.dap.middleware.dmc.event.ProgressEventType;
import com.digiwin.dap.middleware.dmc.event.ProgressListener;
import com.digiwin.dap.middleware.dmc.event.ProgressPublisher;
import com.digiwin.dap.middleware.dmc.internal.DMCConstants;
import com.digiwin.dap.middleware.dmc.internal.DMCOperation;
import com.digiwin.dap.middleware.dmc.internal.DMCRequestMessageBuilder;
import com.digiwin.dap.middleware.dmc.internal.ResponseParsers;
import com.digiwin.dap.middleware.dmc.model.*;

import java.io.*;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * upload process sdk as client,dmc as server,sdk encrypts files through bucket key pair
 *
 * @author michael
 * @since 5.4.0.0
 */
public class DMCUploadEncryptionOperation extends DMCOperation {

    private final DigitalEnvelopeBuilder digitalEnvelopeBuilder;
    private final EncryptionStrategyFactory encryptionStrategyFactory;

    public DMCUploadEncryptionOperation(ServiceClient client,
                                        EncryptionStrategyFactory encryptionStrategyFactory,
                                        DigitalEnvelopeBuilder digitalEnvelopeBuilder) {
        super(client);
        this.encryptionStrategyFactory = encryptionStrategyFactory;
        this.digitalEnvelopeBuilder = digitalEnvelopeBuilder;
    }

    public FileInfo upload(UploadRequest original) throws Exception {
        original.setBytes(this.encryptionStrategyFactory.strategy(original.getBytes()));
        original.setFile(this.encryptionStrategyFactory.strategy((original.getFile())));
        original.setInput(this.encryptionStrategyFactory.strategy((original.getInput())));
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setInputSize(original.getEntity().getContentLength())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .setDigitalEnvelope(digitalEnvelopeBuilder.digitalEnvelope(original.getPublicKey()))
                .setDMCSecurity(Boolean.TRUE)
                .build();
      FileInfo fileInfo = doOperation(request, ResponseParsers.uploadResponseParser, true);
      deleteTempFile(original.getFile());
      return fileInfo;
    }

    public FileInfo coverUpload(UploadCoverRequest original) throws Exception {
        original.setBytes(this.encryptionStrategyFactory.strategy(original.getBytes()));
        original.setFile(this.encryptionStrategyFactory.strategy((original.getFile())));
        original.setInput(this.encryptionStrategyFactory.strategy((original.getInput())));
        RequestMessage request = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadCoverUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setInputSize(original.getEntity().getContentLength())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .setDigitalEnvelope(digitalEnvelopeBuilder.digitalEnvelope(original.getPublicKey()))
                .setDMCSecurity(Boolean.TRUE)
                .build();
        FileInfo fileInfo = doOperation(request, ResponseParsers.uploadResponseParser, true);
        deleteTempFile(original.getFile());
        return fileInfo;
    }

    public List<FileInfo> uploadMulti(UploadMultiRequest original) throws Exception {
        List<File> encryptionFileList = new ArrayList<>();
        for (File file : original.getFiles()) {
            encryptionFileList.add(this.encryptionStrategyFactory.strategy(file));
        }
        List<byte[]> encryptionByteList = new ArrayList<>();
        for (byte[] bytes : original.getBytes()) {
            encryptionByteList.add(this.encryptionStrategyFactory.strategy(bytes));
        }
        List<InputStream> encryptionInputStreamList = new ArrayList<>();
        for (InputStream inputStream : original.getInputs()) {
            encryptionInputStreamList.add(this.encryptionStrategyFactory.strategy(inputStream));
        }
        original.setInputs(encryptionInputStreamList);
        original.setBytes(encryptionByteList);
        original.setFiles(encryptionFileList);

        RequestMessage requestMessage = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadMultiUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setInputSize(original.getEntity().getContentLength())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .setDigitalEnvelope(digitalEnvelopeBuilder.digitalEnvelope(original.getPublicKey()))
                .setDMCSecurity(Boolean.TRUE)
                .build();
        List<FileInfo> fileInfoList = doOperation(requestMessage, ResponseParsers.uploadMultiResponseParser);
        for (File file : original.getFiles()) {
            deleteTempFile(file);
        }
        return fileInfoList;
    }

    public String multipartUpload(MultipartUploadRequest request) throws Exception {
        final long partSize = request.getPartSize();
        if (partSize % DMCConstants.DEFAULT_CHUNK_SIZE != 0) {
            throw new DMCException("分段大小请设置261120的整数倍");
        }
        /*
         * Claim a file id firstly
         */
        String fileId = claimFileId(request.getFileInfo());
        logger.info("Claiming a new file id {}.", fileId);

        long fileLength = request.getFileLength();
        int partCount = (int) (fileLength / partSize);
        if (fileLength % partSize != 0) {
            partCount++;
        }
        if (partCount > 10000) {
            throw new DMCException("分段总数不能超过10000");
        } else {
            logger.info("Total parts count {}.", partCount);
        }

        final ProgressListener listener = request.getProgressListener();
        ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_STARTED_EVENT);
        ProgressPublisher.publishRequestContentLength(listener, fileLength);

        if (request.getInput() != null) {
            try (InputStream inputStream = request.getInput()) {
                for (int i = 0; i < partCount; i++) {
                    long startPos = i * partSize;
                    int curPartSize = (int) ((i + 1 == partCount) ? (fileLength - startPos) : partSize);

                    byte[] buffer = new byte[curPartSize];
                    inputStream.read(buffer);

                    uploadPart(fileId, buffer, startPos, curPartSize, fileLength, partCount, i + 1, listener, request.getPublicKey());
                }
            } catch (Exception e) {
                ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_FAILED_EVENT);
                throw e;
            }
        } else {
            try (RandomAccessFile raf = new RandomAccessFile(request.getFile(), "r")) {
                MappedByteBuffer map = raf.getChannel().map(FileChannel.MapMode.READ_ONLY, 0, fileLength);
                for (int i = 0; i < partCount; i++) {
                    long startPos = i * partSize;
                    int curPartSize = (int) ((i + 1 == partCount) ? (fileLength - startPos) : partSize);

                    byte[] buffer = new byte[curPartSize];
                    map.get(buffer, 0, curPartSize);

                    uploadPart(fileId, buffer, startPos, curPartSize, fileLength, partCount, i + 1, listener, request.getPublicKey());
                }
            } catch (Exception e) {
                ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_FAILED_EVENT);
                throw e;
            }
        }
        ProgressPublisher.publishProgress(listener, ProgressEventType.TRANSFER_COMPLETED_EVENT);
        return fileId;
    }

    private void uploadPart(String fileId, byte[] buffer, long startPos, int curPartSize, long fileLength,
                            int partCount, int partNum, ProgressListener listener, String publicKey) throws Exception {
        PartUploader partUploader = new PartUploader();
        partUploader.setFileId(fileId);
        partUploader.setParts(buffer);
        partUploader.setFrom(startPos);
        partUploader.setTo(startPos + curPartSize - 1);
        partUploader.setTotal(fileLength);
        partUploader.setPublicKey(publicKey);
        partUploader.setPartCount(partCount);
        partUploader.setPartNum(partNum);
        uploadPart(partUploader);

        ProgressPublisher.publishRequestBytesTransferred(listener, curPartSize);
        logger.info("总共：{}段, 第{}段上传成功：fileId={},startPos={},curPartSize={}.", partCount, partNum, fileId, startPos, curPartSize);
    }

    private void uploadPart(PartUploader original) throws Exception {
        byte[] ciphertext = this.encryptionStrategyFactory.strategy(original.getParts());
        original.setParts(ciphertext);
        RequestMessage requestMessage = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadMultipartUrl(original.getFileId(), original.getFrom(), original.getTo(), original.getTotal(), original.getPartNum()))
                .setMethod(HttpMethod.POST)
                .setEntity(original.getEntity())
                .setForceRetry(true)
                .setDigitalEnvelope(digitalEnvelopeBuilder.digitalEnvelope(original.getPublicKey()))
                .setDMCSecurity(Boolean.TRUE)
                .build();
        doOperation(requestMessage, ResponseParsers.uploadMultipartResponseParser);
    }

    private String claimFileId(FileInfo fileInfo) throws Exception {
        FileInfoRequest original = new FileInfoRequest();
        original.setFileInfo(fileInfo);

        RequestMessage requestMessage = DMCRequestMessageBuilder.create()
                .setEndpoint(config.getUploadEmptyUrl())
                .setMethod(HttpMethod.POST)
                .setHeaders(original.getHeaders())
                .setEntity(original.getEntity())
                .setForceRetry(true)
                .setOriginalRequest(original)
                .build();
        return doOperation(requestMessage, ResponseParsers.uploadMultipartResponseParser);
    }

    private void deleteTempFile(File file) {
        if(Objects.nonNull(file) && file.exists()) {
            file.delete();
            logger.info("临时文件删除成功!");
        }
    }

}
