package com.digiwin.dap.middleware.dmc;

import com.digiwin.dap.middleware.dmc.internal.client.OSS;
import com.digiwin.dap.middleware.dmc.model.*;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * Entry point interface of DAP Middleware DMC (Document Management Center)
 * <p>
 * Document Management Center (a.k.a DMC) is the massive, secure, low cost and highly
 * reliable public storage which could be accessed from anywhere at anytime via
 * REST APIs, SDKs. <br>
 * Developers could use DMC to create any services that need huge data storage
 * and access throughput, such as media sharing web apps, cloud storage service
 * or enterprise or personal data backup.
 * </p>
 *
 * @author fobgochod
 * @since 3.1.0.0
 */
public interface DMC extends OSS, SystemConfig {

    /**
     * 复制一个文件到根目录
     *
     * @param fileId 被复制的文件ID
     * @return 新文件ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String copy(String fileId) throws Exception;

    /**
     * 复制文件到一个指定目录
     *
     * @param fileId      被复制的文件ID
     * @param targetDirId 目标目录ID
     * @return 新文件ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String copy(String fileId, String targetDirId) throws Exception;

    /**
     * 复制目录到根目录
     *
     * @param dirId 被复制的目录ID
     * @return 新目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String copyDir(String dirId) throws Exception;

    /**
     * 复制目录到一个指定目录
     *
     * @param dirId       被复制的目录ID
     * @param targetDirId 目标目录ID
     * @return 新目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String copyDir(String dirId, String targetDirId) throws Exception;

    /**
     * 批量复制
     *
     * @param fileIds     被复制的文件ID集合
     * @param dirIds      被复制的目录ID集合
     * @param targetDirId 目标目录ID
     * @return 复制信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileOpTree> batchCopy(List<String> fileIds, List<String> dirIds, String targetDirId) throws Exception;

    /**
     * 批量复制
     *
     * @param request 请求信息
     * @return 复制信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileOpTree> batchCopy(FileOpRequest request) throws Exception;

    /**
     * 剪切文件到根目录
     *
     * @param fileId 被剪切的文件ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void move(String fileId) throws Exception;

    /**
     * 剪切文件到一个指定目录
     *
     * @param fileId      被剪切的文件ID
     * @param targetDirId 目标目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void move(String fileId, String targetDirId) throws Exception;

    /**
     * 剪切目录到根目录
     *
     * @param dirId 被剪切的目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void moveDir(String dirId) throws Exception;

    /**
     * 剪切目录到一个指定目录
     *
     * @param dirId       被剪切的目录ID
     * @param targetDirId 目标目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void moveDir(String dirId, String targetDirId) throws Exception;

    /**
     * 批量剪切
     *
     * @param fileIds     被剪切的文件ID集合
     * @param dirIds      被剪切的目录ID集合
     * @param targetDirId 目标目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchMove(List<String> fileIds, List<String> dirIds, String targetDirId) throws Exception;

    /**
     * 批量剪切
     *
     * @param request 请求信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchMove(FileOpRequest request) throws Exception;

    /**
     * 永久删除指定文件
     *
     * @param fileId 被删除的文件ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void delete(String fileId) throws Exception;

    /**
     * 永久删除指定文件
     *
     * @param dirId 被删除的目录ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void deleteDir(String dirId) throws Exception;

    /**
     * 永久删除文件、目录
     *
     * @param fileIds 被删除的文件ID集合
     * @param dirIds  被删除的目录ID集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchDelete(List<String> fileIds, List<String> dirIds) throws Exception;

    /**
     * 永久删除文件、目录
     *
     * @param request 请求信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchDelete(BatchRequest request) throws Exception;

    /**
     * 普通上传
     *
     * @param file file
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(File file) throws Exception;

    /**
     * 普通上传
     *
     * @param file  文件
     * @param dirId 目录、
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(File file, String dirId) throws Exception;

    /**
     * 普通上传
     *
     * @param file     文件
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     * @param dirId    目录
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    FileInfo upload(File file, String filename, String dirId) throws Exception;

    /**
     * 普通上传，指定文件信息
     *
     * @param file     文件
     * @param fileInfo 文件信息
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(File file, FileInfo fileInfo) throws Exception;

    /**
     * 普通上传
     *
     * @param bytes    bytes 字节数组
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(byte[] bytes, String filename) throws Exception;

    /**
     * 普通上传
     *
     * @param bytes    bytes 字节数组
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     * @param dirId    目录ID
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    FileInfo upload(byte[] bytes, String filename, String dirId) throws Exception;

    /**
     * 普通上传，指定文件信息
     *
     * @param bytes    字节数组
     * @param fileInfo 文件信息
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(byte[] bytes, FileInfo fileInfo) throws Exception;

    /**
     * 普通上传
     *
     * @param input    input 输入流
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(InputStream input, String filename) throws Exception;

    /**
     * 普通上传
     *
     * @param input    input 输入流
     * @param filename 文件名 带后缀名 <span>VanGogh.jpg</span>
     * @param dirId    目录ID
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    FileInfo upload(InputStream input, String filename, String dirId) throws Exception;

    /**
     * 普通上传，指定文件信息
     *
     * @param input    输入流
     * @param fileInfo 文件信息
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(InputStream input, FileInfo fileInfo) throws Exception;

    /**
     * 普通上传
     *
     * @param request 请求信息
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo upload(UploadRequest request) throws Exception;

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param file   file
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.1.0.0
     */
    FileInfo coverUpload(String fileId, File file) throws Exception;

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param bytes  字节数组
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.1.0.0
     */
    FileInfo coverUpload(String fileId, byte[] bytes) throws Exception;

    /**
     * 覆盖上传
     *
     * @param fileId 文件ID
     * @param input  输入流
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.1.0.0
     */
    FileInfo coverUpload(String fileId, InputStream input) throws Exception;

    /**
     * 批量上传
     *
     * @param files 文件集合
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUpload(List<File> files) throws Exception;

    /**
     * 批量上传
     * 文件、文件信息一一对应
     *
     * @param files     文件集合
     * @param fileInfos 文件信息集合
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUpload(List<File> files, List<FileInfo> fileInfos) throws Exception;

    /**
     * 批量上传
     *
     * @param bytes     字节数组集合
     * @param filenames 文件名集合
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUploadBytes(List<byte[]> bytes, List<String> filenames) throws Exception;

    /**
     * 批量上传
     *
     * @param bytes     字节数组集合
     * @param fileInfos 文件名集合
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUploadBytes(List<byte[]> bytes, Map<String, FileInfo> fileInfos) throws Exception;

    /**
     * 批量上传
     *
     * @param inputs    输入流集合
     * @param filenames 文件名集合
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUploadStream(List<InputStream> inputs, List<String> filenames) throws Exception;

    /**
     * 批量上传
     *
     * @param inputs    输入流集合
     * @param fileInfos 文件信息集合
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUploadStream(List<InputStream> inputs, Map<String, FileInfo> fileInfos) throws Exception;

    /**
     * 批量上传
     *
     * @param request 请求信息
     * @return 文件信息集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<FileInfo> batchUpload(UploadMultiRequest request) throws Exception;

    /**
     * 分段上传
     *
     * @param file 文件
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(File file) throws Exception;

    /**
     * 分段上传
     *
     * @param file     文件
     * @param partSize 分段大小
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(File file, int partSize) throws Exception;

    /**
     * 分段上传
     *
     * @param file     文件
     * @param fileInfo 文件信息
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(File file, FileInfo fileInfo) throws Exception;

    /**
     * 分段上传
     *
     * @param file     文件
     * @param fileInfo 文件信息
     * @param partSize 分段大小
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(File file, FileInfo fileInfo, int partSize) throws Exception;

    /**
     * 分段上传
     *
     * @param input    输入流
     * @param fileInfo 文件信息  fileName、size必填
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(InputStream input, FileInfo fileInfo) throws Exception;

    /**
     * 分段上传
     *
     * @param input    输入流
     * @param fileInfo 文件信息 fileName、size必填
     * @param partSize 分段大小
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(InputStream input, FileInfo fileInfo, int partSize) throws Exception;

    /**
     * 分段上传
     *
     * @param request 请求信息
     * @return 文件ID(fileId)
     * @throws Exception 程序处理过程遇到的任何错误
     */
    String multipartUpload(MultipartUploadRequest request) throws Exception;

    /**
     * 下载指定文件
     *
     * @param fileId 被下载的文件ID
     * @return 文集字节码
     * @throws Exception 程序处理过程遇到的任何错误
     */
    byte[] download(String fileId) throws Exception;

    /**
     * 下载指定文件到本地
     *
     * @param fileId   被下载的文件ID
     * @param filePath 文件路径
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    void download(String fileId, String filePath) throws Exception;

    /**
     * 下载指定文件到本地
     *
     * @param fileId   被下载的文件ID
     * @param filePath 文件路径
     * @param fileName 文件名
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    void download(String fileId, String filePath, String fileName) throws Exception;

    /**
     * 下载指定文件到流
     *
     * @param fileId 被下载的文件ID
     * @param output 输出流
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void download(String fileId, OutputStream output) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param fileIds 被下载的文件ID集合
     * @param dirIds  被下载的目录ID集合
     * @return 文集字节码
     * @throws Exception 程序处理过程遇到的任何错误
     */
    byte[] batchDownload(List<String> fileIds, List<String> dirIds) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param fileIds  被下载的文件ID集合
     * @param dirIds   被下载的目录ID集合
     * @param filePath 文件路径
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    void batchDownload(List<String> fileIds, List<String> dirIds, String filePath) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param fileIds  被下载的文件ID集合
     * @param dirIds   被下载的目录ID集合
     * @param filePath 文件路径
     * @param fileName 文件名(xxx.zip)
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    void batchDownload(List<String> fileIds, List<String> dirIds, String filePath, String fileName) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param fileIds 被下载的文件ID集合
     * @param dirIds  被下载的目录ID集合
     * @param output  输出流
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchDownload(List<String> fileIds, List<String> dirIds, OutputStream output) throws Exception;

    /**
     * 批量下载(返回.zip文件)
     *
     * @param request 请求信息
     * @return 文集字节码
     * @throws Exception 程序处理过程遇到的任何错误
     */
    byte[] batchDownload(BatchRequest request) throws Exception;

    /**
     * 根据图片ID获取缩放图片
     *
     * @param fileId 原图文件ID
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<ImageInfo> getImageInfo(String fileId) throws Exception;

    /**
     * 根据图片ID获取缩放图片
     *
     * @param fileId 原图文件ID
     * @param resize false-原图 true-缩放后的图片
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<ImageInfo> getImageInfo(String fileId, boolean resize) throws Exception;

    /**
     * 指定修改图片宽度，高度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageInfo resizeImageWidth(String fileId, int width) throws Exception;

    /**
     * 指定修改图片宽度，高度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @param share  是否分享
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageInfo resizeImageWidth(String fileId, int width, boolean share) throws Exception;

    /**
     * 指定修改图片高度，宽度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param height 要设定的高度
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageInfo resizeImageHeight(String fileId, int height) throws Exception;

    /**
     * 指定修改图片高度，宽度等比例缩放
     *
     * @param fileId 原图文件ID
     * @param height 要设定的高度
     * @param share  是否分享
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageInfo resizeImageHeight(String fileId, int height, boolean share) throws Exception;

    /**
     * 根据图片ID缩放图片
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @param height 要设定的高度
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageInfo resizeImage(String fileId, int width, int height) throws Exception;

    /**
     * 根据图片ID缩放图片
     *
     * @param fileId 原图文件ID
     * @param width  要设定的宽度
     * @param height 要设定的高度
     * @param share  是否分享
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageInfo resizeImage(String fileId, int width, int height, boolean share) throws Exception;

    /**
     * 图片上传
     *
     * @param image     image
     * @param imageInfo 图片信息
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageUpload resizeImage(File image, FileInfo imageInfo) throws Exception;

    /**
     * 图片上传 指定缩放宽度
     *
     * @param image     image
     * @param imageInfo 文件信息
     * @param width     要设定的宽度
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageUpload resizeImageWidth(File image, FileInfo imageInfo, int width) throws Exception;

    /**
     * 图片上传 指定缩放高度
     *
     * @param image     image
     * @param imageInfo 文件信息
     * @param height    要设定的高度
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageUpload resizeImageHeight(File image, FileInfo imageInfo, int height) throws Exception;

    /**
     * 图片上传并缩放
     *
     * @param image     image
     * @param imageInfo 文件信息
     * @param width     要设定的宽度
     * @param height    要设定的高度
     * @return 图片信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ImageUpload resizeImage(File image, FileInfo imageInfo, int width, int height) throws Exception;

    /**
     * 删除文件到回收站
     *
     * @param fileId 被删除的文件ID
     * @return 回收站信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    RecycleBin remove(String fileId) throws Exception;

    /**
     * 删除目录到回收站
     *
     * @param dirId 被删除的目录ID
     * @return 回收站信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    RecycleBin removeDir(String dirId) throws Exception;

    /**
     * 批量删除文件、目录到回收站
     *
     * @param fileIds 被删除的文件ID集合
     * @param dirIds  被删除的目录ID集合
     * @return 回收站信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<RecycleBin> batchRemove(List<String> fileIds, List<String> dirIds) throws Exception;

    /**
     * 批量删除文件、目录到回收站
     *
     * @param request 请求信息
     * @return 回收站信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    List<RecycleBin> batchRemove(BatchRequest request) throws Exception;

    /**
     * 删除回收站指定文件
     *
     * @param recycleId 回收站ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void deleteRecycleBin(String recycleId) throws Exception;

    /**
     * 批量删除回收站指定文件
     *
     * @param recycleIds 回收站ID集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchDeleteRecycleBin(List<String> recycleIds) throws Exception;

    /**
     * 清空回收站
     *
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void clearRecycleBin() throws Exception;

    /**
     * 恢复回收站文件
     *
     * @param recycleId 回收站ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void restoreRecycleBin(String recycleId) throws Exception;

    /**
     * 批量恢复回收站文件
     *
     * @param recycleIds 回收站ID集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchRestoreRecycleBin(List<String> recycleIds) throws Exception;

    /**
     * 批量恢复回收站文件
     *
     * @param request 请求信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchRestoreRecycleBin(BatchRequest request) throws Exception;

    /**
     * 恢复回收站文件
     *
     * @param fileId 被删除的文件ID
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.3.2.0
     */
    void restore(String fileId) throws Exception;

    /**
     * 恢复回收站文件
     *
     * @param fileIds 被删除的文件ID集合
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.3.2.0
     */
    void batchRestore(List<String> fileIds) throws Exception;

    /**
     * 分享指定文件
     *
     * @param fileId 被分享的文件ID
     * @return 分享信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    ShareInfo share(String fileId) throws Exception;

    /**
     * 分享指定文件，指定分享链接过期时间
     *
     * @param fileId     被分享的文件ID
     * @param expireDate 过期使时间
     * @return 分享信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.3.2.0
     */
    ShareInfo share(String fileId, LocalDateTime expireDate) throws Exception;

    /**
     * 分享指定文件，指定分享链接可使用次数
     *
     * @param fileId     被分享的文件ID
     * @param allowCount 分享地址可使用次数
     * @return 分享信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.3.2.0
     */
    ShareInfo share(String fileId, int allowCount) throws Exception;

    /**
     * 分享指定文件，指定分享链接可使用次数，并指定是否在次数用完后删除文件
     *
     * @param fileId     被分享的文件ID
     * @param allowCount 分享地址可使用次数
     * @param deleteFile 分享次数为0时，是否删除文件
     * @return 分享信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.3.2.0
     */
    ShareInfo share(String fileId, int allowCount, boolean deleteFile) throws Exception;

    /**
     * 分享指定文件
     *
     * @param request 分享信息
     * @return 分享信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.3.2.0
     */
    ShareInfo share(ShareRequest request) throws Exception;

    /**
     * 删除分享记录
     *
     * @param shareId 分享ID
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void deleteShare(String shareId) throws Exception;

    /**
     * 批量删除分享记录
     *
     * @param shareIds 分享ID集合
     * @throws Exception 程序处理过程遇到的任何错误
     */
    void batchDeleteShare(List<String> shareIds) throws Exception;


    /**********************************************************************
     *                                                                    *
     *                             目录基本信息操作                          *
     *                                                                    *
     **********************************************************************/

    /**
     * 创建目录
     *
     * @param dirInfo 目录信息
     * @return 目录信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    DirInfo createDir(DirInfo dirInfo) throws Exception;

    /**
     * 重命名文件
     *
     * @param dirId   文件ID
     * @param dirName 新文件名
     * @return 目录信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    DirInfo renameDir(String dirId, String dirName) throws Exception;

    /**
     * 修改目录信息
     *
     * @param dirInfo 目录信息
     * @return 目录信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    DirInfo modDirInfo(DirInfo dirInfo) throws Exception;

    /**
     * 获取目录信息
     *
     * @param dirId 目录ID
     * @return 目录信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    DirInfo getDirInfo(String dirId) throws Exception;

    /**
     * 通过目录名称或者路径获取目录
     *
     * @param dirName 目录名称或者路径
     * @return 目录信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.3.0
     */
    DirInfo getDirInfoByName(String dirName) throws Exception;

    /**
     * 通过目录名称或者路径获取目录.
     *
     * <table summary="目录结构">
     *     <tr><th>目录ID</th><th>父目录ID</th><th>目录名称</th></tr>
     *     <tr><td>0001</td><td>0000</td><td>一级目录</td></tr>
     *     <tr><td>0002</td><td>0001</td><td>二级目录</td></tr>
     *     <tr><td>0003</td><td>0002</td><td>三级目录</td></tr>
     * </table>
     * <p>
     * <b>范例：查询三级目录</b>
     * <table summary="查询方案">
     *     <tr><th>parentDirId</th><th>dirName</th></tr>
     *     <tr><td>0002</td><td>三级目录</td></tr>
     *     <tr><td>0001</td><td>二级目录/三级目录</td></tr>
     *     <tr><td>0000</td><td>一级目录/二级目录/三级目录</td></tr>
     * </table>
     * 注：范例中的ID已经简化，实际ID是一个UUID字符串，查询根下的目录parentDirId可不传
     *
     * @param parentDirId 父目录ID
     * @param dirName     目录名称或者路径
     * @return 目录信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.3.0
     */
    DirInfo getDirInfoByName(String parentDirId, String dirName) throws Exception;


    /**********************************************************************
     *                                                                    *
     *                             文件基本信息操作                          *
     *                                                                    *
     **********************************************************************/

    /**
     * 重命名文件
     *
     * @param fileId   文件ID
     * @param fileName 新文件名
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.4.0
     */
    FileInfo renameFile(String fileId, String fileName) throws Exception;

    /**
     * 修改文件信息
     *
     * @param fileInfo 文件信息
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo modFileInfo(FileInfo fileInfo) throws Exception;

    /**
     * 设置文件过期时间
     *
     * @param fileId     文件ID
     * @param expireDate 过期时间
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo modFileExpireDate(String fileId, LocalDateTime expireDate) throws Exception;

    /**
     * 修改文件目录
     *
     * @param fileId 文件ID
     * @param dirId  目录ID
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo modFileDirectoryId(String fileId, String dirId) throws Exception;

    /**
     * 查询文件信息
     *
     * @param fileId 文件ID
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     */
    FileInfo getFileInfo(String fileId) throws Exception;

    /**
     * 查询文件信息
     *
     * @param fileName 文件名
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.3.0
     */
    FileInfo getFileInfoByName(String fileName) throws Exception;

    /**
     * 查询文件信息
     *
     * @param dirId    目录ID
     * @param fileName 文件名
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.3.0
     */
    FileInfo getFileInfoByName(String dirId, String fileName) throws Exception;

    /**
     * 查询目录下的文件信息
     *
     * @param dirId 目录ID
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.3.0
     */
    List<FileInfo> getFileInfos(String dirId) throws Exception;

    /**
     * 查询指定IDs的文件信息
     *
     * @param fileIds 文件ID集合
     * @return 文件信息
     * @throws Exception 程序处理过程遇到的任何错误
     * @since 5.2.3.0
     */
    List<FileInfo> getFileInfos(List<String> fileIds) throws Exception;

    /**
     * 获取系统账号Token
     *
     * @return token
     */
    String getToken();

    /**
     * 获取指定租户对应的系统账号Token
     *
     * @param tenantId 租户ID
     * @return token
     */
    String getToken(String tenantId);

    /**
     * 切换登录租户到默认(default)租户
     * <p>
     * 无租户即default租户，可以操作所有租户文件
     */
    void switchTenantId();

    /**
     * Switches to another tenantId
     * <p>
     * 切换登录租户，当前client生效
     * <ol>
     *     <li>上传文件均归属该租户</li>
     *     <li>没有权限操作其他租户的文件</li>
     *     <li>和{@link #setTenantId(String)}不要混用，调用后会使{@link #setTenantId(String)}失效</li>
     * </ol>
     *
     * @param tenantId the tenantId to switch to。
     */
    void switchTenantId(String tenantId);

    /**
     * 设置上传租户，当前线程生效
     *
     * <ol>
     *     <li>只有默认(default)租户才能使用</li>
     *     <li>文件都上传到该租户</li>
     *     <li>本质是更新fileInfo一个属性tenantId</li>
     *     <li>和{@link #switchTenantId()}不要混用，切换租户，该设定会被清空</li>
     * </ol>
     *
     * @param tenantId 租户ID
     * @return DMC客户端
     * @since 5.1.0.0
     */
    DMC setTenantId(String tenantId);

    /**
     * 设置Iam用户Token，当前线程生效
     *
     * @param iamUserToken Iam用户token
     * @return DMC客户端
     * @since 5.4.0.0
     */
    DMC setIamUserToken(String iamUserToken);

    /**
     * 获取Iam用户Token，当前线程仅一次有效，调用dmc接口后会自动清除
     *
     * @param iamUserToken Iam用户token
     * @param useOnce      是否一次性使用，为true每次调用dmc后自动清除线程变量
     * @return Iam用户Token
     * @since 5.4.0.0
     */
    DMC setIamUserToken(String iamUserToken, boolean useOnce);

    /**
     * 清除当前线程Iam用户Token
     */
    void clearIamUserToken();

    /**
     * Shuts down the DMC instance (release all resources) The DMC instance is
     * not usable after its shutdown() is called.
     */
    void shutdown();

    /**
     * Get the statistics of the connection pool.
     *
     * @return A status in string format.
     */
    String getConnectionPoolStats();
}
