package com.digiwin.dap.middleware.lmc.http.comm;

import com.digiwin.dap.middleware.lmc.common.Consts;
import com.digiwin.dap.middleware.lmc.common.auth.AuthCache;
import com.digiwin.dap.middleware.lmc.http.client.ClientConfiguration;
import com.digiwin.dap.middleware.lmc.http.client.ClientException;
import com.digiwin.dap.middleware.lmc.http.client.ServiceException;
import com.digiwin.dap.middleware.lmc.http.utils.CodingUtils;
import com.digiwin.dap.middleware.lmc.http.utils.HttpUtil;
import com.digiwin.dap.middleware.lmc.util.LMCUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;

/**
 * Abstract service client that provides interfaces to access LMC services.
 *
 * @author chenzhuang
 * @date 2021/4/25
 */
public abstract class ServiceClient {

    protected static final Logger logger = LoggerFactory.getLogger(ServiceClient.class);
    protected ClientConfiguration config;

    protected ServiceClient(ClientConfiguration config) {
        this.config = config;
    }

    public ClientConfiguration getClientConfiguration() {
        return this.config;
    }

    protected abstract ResponseMessage sendRequestCore(RequestMessage request) throws IOException;

    public abstract AuthCache getAuthCache();

    public abstract void setAuthCache(String endpoint, String token);

    public abstract void shutdown();

    /**
     * Send HTTP request with specified context to OSS and wait for HTTP
     * response.
     */
    public ResponseMessage sendRequest(RequestMessage request) throws ServiceException, ClientException {
        CodingUtils.assertParameterNotNull(request, "request");

        try {
            return sendRequestImpl(request);
        } finally {
            // Close the request stream as well after the request is completed.
            try {
                request.close();
            } catch (IOException ex) {
                logger.error("Unexpected io exception when trying to close http request: ", ex);
                throw new ClientException("Unexpected io exception when trying to close http request: ", ex);
            }
        }
    }

    public ResponseMessage sendRequestImpl(RequestMessage request) throws ServiceException, ClientException {
        ResponseMessage response = null;
        try {
            buildRequest(request, Consts.UTF_8_ENCODING);
            // Send HTTP request to LMC.
            return sendRequestCore(request);
        } catch (ServiceException ex) {
            logger.error("[Server]Unable to execute HTTP request: ", ex);
            throw ex;
        } catch (ClientException ex) {
            logger.error("[Client]Unable to execute HTTP request: ", ex);
            throw ex;
        } catch (Exception ex) {
            logger.error("[Unknown]Unable to execute HTTP request: ", ex);
            throw new ClientException(LMCUtils.COMMON_RESOURCE_MANAGER.getFormattedString("ConnectionError", ex.getMessage()), ex);
        }
    }

    private RequestMessage buildRequest(RequestMessage request, String charset) throws ClientException {
        String paramString = HttpUtil.paramToQueryString(request.getParameters(), charset);
        String uri = request.getEndpoint().toString();
        if (paramString != null) {
            uri += "?" + paramString;
        }
        request.setEndpoint(URI.create(uri));
        return request;
    }
}


