/*
 * Copyright (C) Alibaba Cloud Computing All rights reserved.
 */
package com.digiwin.dap.middleware.lmc;


import com.digiwin.dap.middleware.lmc.internal.model.ModifiedInfo;
import com.digiwin.dap.middleware.lmc.internal.model.PageData;
import com.digiwin.dap.middleware.lmc.internal.model.StdData;
import com.digiwin.dap.middleware.lmc.internal.model.entity.*;
import com.digiwin.dap.middleware.lmc.request.*;

import java.util.List;
import java.util.Map;

/**
 * 日志中心Service
 */
public interface LMC {
    /**
     * 添加操作日志
     * <p>
     * 直接保存操作日志的变更结果
     *
     * @param saveOpLog 保存的操作日志；
     *                  必填：moduleId、actId、operationType;
     *                  变更结果[changeInfoList]
     *                  填入字段[key]、字段名称[displayName]、字段的原始值[originValue]和修改值[modifyValue]
     * @param userToken 用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveOpLog(SaveOpLog saveOpLog, String userToken) throws Exception;

    /**
     * 添加操作日志
     * <p>
     * 直接保存操作日志的变更结果
     *
     * @param saveOpLog      保存的操作日志；
     *                       必填：moduleId、actId、operationType;
     * @param changeInfoList 变更结果
     *                       填入字段[key]、字段名称[displayName]、字段的原始值[originValue]和修改值[modifyValue]
     * @param userToken      用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveOpLog(SaveOpLog saveOpLog, List<ChangeInfo> changeInfoList, String userToken) throws Exception;

    /**
     * 根据设定,添加比对的操作日志
     * <p>
     * 根据设定的日志事件模板比对后，保存操作日志
     * 必填：moduleId、actId、operationType;
     *
     * @param saveCompareOpLog 需要保存的操作日志
     * @param userToken        用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveCompareOpLog(SaveCompareOpLog saveCompareOpLog, String userToken) throws Exception;

    /**
     * 根据设定,添加比对的操作日志
     * <p>
     * 根据设定的日志事件模板比对后，保存操作日志
     * 必填：moduleId、actId、operationType;
     *
     * @param saveCompareOpLog 需要保存的操作日志
     * @param originJson       变更前的json
     * @param modifyJson       变更后的json
     * @param userToken        用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveCompareOpLog(SaveCompareOpLog saveCompareOpLog, Map<String, Object> originJson, Map<String, Object> modifyJson, String userToken) throws Exception;

    /**
     * 添加比对的操作日志
     * <p>
     * 必填：moduleId、actId、operationType、primaryKeyList、logFieldList;
     *
     * @param saveCompareOpLogDirect 需要保存的操作日志
     * @param userToken              用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveCompareOpLogDirect(SaveCompareOpLog saveCompareOpLogDirect, String userToken) throws Exception;

    /**
     * 添加比对的操作日志
     * <p>
     * 必填：moduleId、actId、operationType、primaryKeyList、logFieldList;
     *
     * @param saveCompareOpLogDirect 需要保存的操作日志
     * @param originJson             变更前的json
     * @param modifyJson             变更后的json
     * @param primaryKeyList         主键集合
     * @param logFieldList           关注字段名称集合
     * @param userToken              用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveCompareOpLogDirect(SaveCompareOpLog saveCompareOpLogDirect, Map<String, Object> originJson, Map<String, Object> modifyJson, List<Map<String, String>> primaryKeyList, List<Map<String, String>> logFieldList, String userToken) throws Exception;

    /**
     * 分页查询操作日志
     *
     * @param dto       查询条件 OpLogDTO
     *                  选填：appId（默认client.app）
     * @param pageNum   页码
     * @param pageSize  页面大小
     * @param orders    排序，-1倒叙 1正序 例如{createTime : -1, name : 1}
     *                  不传递[默认createDate倒叙]
     * @param userToken 用户userToken
     * @return PageData<OpLog> 返回查询结果
     */
    PageData<OpLog> findOpLogByPage(OpLogDTO dto, int pageNum, int pageSize, Map<String, Object> orders, String userToken) throws Exception;

    /**
     * 分页查询操作日志
     *
     * @param request   查询条件
     *                  选填：opLogDTO.appId（默认client.app）
     *                  pageNum  页码
     *                  pageSize 页面大小
     *                  orders   排序，-1倒叙 1正序 例如{createTime : -1, name : 1}
     *                  不传递[默认createDate倒叙]
     * @param userToken 用户userToken
     * @return PageData<OpLog> 返回查询结果
     */
    PageData<OpLog> findOpLogByPage(PageOpLogRequest request, String userToken) throws Exception;

    /**
     * 添加系统日志
     *
     * @param saveDevLog 保存的系统日志；
     *                     必填：level、message;
     * @return DevLog 返回保存结果
     */
    DevLog saveDevLog(SaveDevLog saveDevLog) throws Exception;

    /**
     * 添加系统日志
     *
     * @param saveDevLog 保存的操作日志；
     *                     必填：level、message;
     * @param userToken    [选填]用户userToken
     * @return DevLog 返回保存结果
     */
    DevLog saveDevLog(SaveDevLog saveDevLog, String userToken) throws Exception;

    /**
     * 分页查询系统日志
     *
     * @param dto       查询条件 DevLogDTO
     *                  选填：appId（默认client.app）
     * @param pageNum   页码
     * @param pageSize  页面大小
     * @param orders    排序，-1倒叙 1正序 例如{time : -1, name : 1}
     *                  不传递[默认createDate倒叙]
     * @param userToken 用户userToken
     * @return PageData<DevLog> 返回查询结果
     */
    PageData<DevLog> findDevLogByPage(DevLogDTO dto, int pageNum, int pageSize, Map<String, Object> orders, String userToken) throws Exception;

    /**
     * 分页查询系统日志
     *
     * @param request   查询条件
     *                  选填：opLogDTO.appId（默认client.app）
     *                  pageNum  页码
     *                  pageSize 页面大小
     *                  orders   排序，-1倒叙 1正序 例如{time : -1, name : 1}
     *                  不传递[默认createDate倒叙]
     * @param userToken 用户userToken
     * @return PageData<DevLog> 返回查询结果
     */
    PageData<DevLog> findDevLogByPage(PageDevLogRequest request, String userToken) throws Exception;

    /**
     * 添加事件日志
     *
     * @param saveEventLog 保存的事件日志；
     *                     必填：eventName、eventType;
     * @return EventLog 返回保存结果
     */
    EventLog saveEventLog(SaveEventLog saveEventLog) throws Exception;

    /**
     * 添加事件日志
     *
     * @param saveEventLog 保存的事件日志；
     *                     必填：eventName、eventType;
     * @param userToken    用户userToken
     * @return EventLog 返回保存结果
     */
    EventLog saveEventLog(SaveEventLog saveEventLog, String userToken) throws Exception;

    /**
     * 批量添加事件日志
     *
     * @param saveEventLogList 保存的事件日志集合；
     *                         必填：eventName、eventType;
     * @return EventLog 返回保存结果
     */
    StdData saveEventLogBatch(List<SaveEventLog> saveEventLogList) throws Exception;

    /**
     * 批量添加事件日志
     *
     * @param saveEventLogList 保存的事件日志集合；
     *                         必填：eventName、eventType;
     * @param userToken        用户userToken
     * @return EventLog 返回保存结果
     */
    StdData saveEventLogBatch(List<SaveEventLog> saveEventLogList, String userToken) throws Exception;

    /**
     * 分页查询事件日志
     *
     * @param dto       查询条件 EventLogDTO
     *                  选填：appId（默认client.app）
     * @param pageNum   页码
     * @param pageSize  页面大小
     * @param orders    排序，-1倒叙 1正序 例如{createTime : -1, name : 1}
     *                  不传递[默认createDate倒叙]
     * @param userToken 用户userToken
     * @return PageData<EventLog> 返回查询结果
     */
    PageData<EventLog> findEventLogByPage(EventLogDTO dto, int pageNum, int pageSize, Map<String, Object> orders, String userToken) throws Exception;

    /**
     * 分页查询事件日志
     *
     * @param request   查询条件
     *                  选填：opLogDTO.appId（默认client.app）
     *                  pageNum  页码
     *                  pageSize 页面大小
     *                  orders   排序，-1倒叙 1正序 例如{createTime : -1, name : 1}
     *                  不传递[默认createDate倒叙]
     * @param userToken 用户userToken
     * @return PageData<EventLog> 返回查询结果
     */
    PageData<EventLog> findEventLogByPage(PageEventLogRequest request, String userToken) throws Exception;

    /**
     * 变更信息直接保存到数据库
     * <p>
     * 用于鼎捷云digiwincloud使用原DMC操作历程
     * <p>
     * /api/lmc/v1/buckets/digiwincloud/changelogs/result
     * 使用相同实体类名CompareInfo传递参数
     *
     * @param dto       变更信息
     * @param userToken 用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog saveChangeLogs(CompareInfo dto, String userToken) throws Exception;

    /**
     * 比对变更信息，并保存到数据库
     * <p>
     * 用于鼎捷云digiwincloud使用原DMC操作历程
     * <p>
     * /api/lmc/v1/buckets/digiwincloud/changelogs
     * 使用相同实体类名ModifiedInfo传递参数
     *
     * @param dto       需要比对的变更历程
     * @param userToken 用户userToken
     * @return OpLog 返回保存结果
     */
    OpLog dataComparison(ModifiedInfo dto, String userToken) throws Exception;

    /**
     * Shuts down the LMC instance (release all resources) The LMC instance is
     * not usable after its shutdown() is called.
     */
    void shutdown();
}
