package com.digiwin.dap.middleware.lmc.appender;

import com.digiwin.dap.middleware.lmc.common.TimeingSingletonEnum;
import com.digiwin.dap.middleware.lmc.http.client.HttpConfig;
import com.digiwin.dap.middleware.lmc.internal.LMCResourceUri;
import com.digiwin.dap.middleware.lmc.util.LogUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.apache.logging.log4j.core.util.Throwables;
import org.apache.logging.log4j.util.PropertiesUtil;
import org.json.JSONArray;

import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * DwLogAppender
 *
 * @author chenzhuang
 */
@Plugin(
    name = "DwLog4j2Appender",
    category = "Core",
    elementType = "appender",
    printObject = true
)
public class DwLog4j2Appender extends AbstractAppender {
    static final String PROPERTY_NAME_ASYNC_EVENT_ROUTER = "log4j2.AsyncQueueFullPolicy";
    static final String PROPERTY_VALUE_DISCARDING_ASYNC_EVENT_ROUTER = "Discard";
    /** 间隔时间 mill */
    public static final int PERIOD_MILL_SECONDS = 2 * 1000;
    private final SimpleDateFormat _sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
    /** 批量插入队列size */
    public static final int QUEUE_SIZE = 50;
    private static final Queue<Map<String, Object>> QUEUE = new LinkedBlockingQueue<>(QUEUE_SIZE);
    private CloseableHttpClient client;

    private String endpoint;
    private String app;
    private String userAgent = "log4j2";

    protected DwLog4j2Appender(String name, Filter filter, Layout<? extends Serializable> layout, String endpoint, String app) {
        super(name, filter, layout);
        this.endpoint = endpoint;
        this.app = app;
    }

    @Override
    public void start() {
        super.start();
        // 请求配置
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(HttpConfig.CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(HttpConfig.CONNECT_REQUEST_TIMEOUT)
                .setSocketTimeout(HttpConfig.SOCKET_TIMEOUT).build();

        // 连接池管理
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setMaxTotal(HttpConfig.MAX_TOTAL_CONNECTIONS);
        connectionManager.setDefaultMaxPerRoute(HttpConfig.MAXIMUM_CONNECTION_PER_ROUTE);
        connectionManager.setValidateAfterInactivity(HttpConfig.CONNECTION_VALIDATE_AFTER_INACTIVITY_MS);
        this.client = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager).build();

        initProperties();
        initInsertQueueTask();
    }

    @Override
    public void stop() {
        super.stop();
        if (this.client != null) {
            try {
                this.client.close();
            } catch (IOException e) {
                LOGGER.error(e.getMessage());
            }
        }
    }

    @Override
    public void append(LogEvent event) {
        Map<String, Object> logMap = new HashMap();
        Map<String, String> contextDataMap = event.getContextData().toMap();
        String traceId;
        if (!contextDataMap.isEmpty()) {
            String ptxId = contextDataMap.get("PtxId");
            // String pspanId = contextDataMap.get("PspanId ");
            if (ptxId != null && !"".equals(ptxId)) {
                traceId = ptxId;
            } else {
                traceId = LogUtils.getUUID();
            }
        } else {
            traceId = LogUtils.getUUID();
        }
        logMap.putIfAbsent("traceId", traceId);

        Map<String, Object> initMap = new HashMap<>();
        LogUtils.initLogMap(initMap);
        logMap.putAll(initMap);
        logMap.putIfAbsent("appId", this.getApp());
        logMap.putIfAbsent("time", this._sdf.format(new Date()));
        logMap.putIfAbsent("level", event.getLevel().toString());
        logMap.putIfAbsent("thread", event.getThreadName());

        logMap.putIfAbsent("loggerName", event.getLoggerName());
        String addr = LogUtils.getLocalHostIpName();
        logMap.putIfAbsent("source", addr);

        logMap.putIfAbsent("appender", "DwLog4j2Appender");

        StackTraceElement source = event.getSource();
        if (source == null && (!event.isIncludeLocation())) {
            event.setIncludeLocation(true);
            source = event.getSource();
            event.setIncludeLocation(false);
        }

        logMap.putIfAbsent("location", source == null ? "Unknown(Unknown Source)" : source.toString());

        String message = event.getMessage().getFormattedMessage();
        logMap.putIfAbsent("message", message);

        String throwable = getThrowableStr(event.getThrown());
        if (throwable != null) {
            logMap.putIfAbsent("throwable", throwable);
        }

        if (getLayout() != null) {
            logMap.putIfAbsent("log", new String(getLayout().toByteArray(event)));
        }

        if (QUEUE.offer(logMap)) {
            return;
        }
        insertQueue(logMap);
    }

    public void insertQueue(Map<String, Object> logMap) {
        if (logMap == null) {
            boolean executable = System.currentTimeMillis() - TimeingSingletonEnum.getInstance().getExecutionTimeMs() > PERIOD_MILL_SECONDS && !QUEUE.isEmpty();
            if (!executable) {
                return;
            }
        }
        List<Map<String, Object>> insertList = new ArrayList<>(QUEUE);
        QUEUE.clear();
        TimeingSingletonEnum.getInstance().setExecutionTimeMs(System.currentTimeMillis());
        Optional.ofNullable(logMap).ifPresent(QUEUE::offer);
        if (insertList.isEmpty()) {
            return;
        }

        CloseableHttpResponse hResponse = null;
        HttpPost postMethod = new HttpPost(LMCResourceUri.getSaveDevLogBatchUrl(this.endpoint));
        try {
            JSONArray tmepObjects = new JSONArray(insertList);
            String formatValue = tmepObjects.toString();
            postMethod.setEntity(new StringEntity(formatValue, ContentType.APPLICATION_JSON));

            hResponse = this.client.execute(postMethod);
            HttpEntity repEntity = hResponse.getEntity();
            int statusCode = hResponse.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                postMethod.abort();
            }
        } catch (Exception e) {
            this.error(e.getMessage());
        } finally {
            if (hResponse != null) {
                try {
                    hResponse.close();
                } catch (IOException e) {
                    this.error(e.getMessage());
                }
            }
        }

    }

    private void initProperties() {
        if (PropertiesUtil.getProperties().getStringProperty(PROPERTY_NAME_ASYNC_EVENT_ROUTER) == null) {
            System.setProperty(PROPERTY_NAME_ASYNC_EVENT_ROUTER, PROPERTY_VALUE_DISCARDING_ASYNC_EVENT_ROUTER);
        }
    }

    private void initInsertQueueTask() {
        ScheduledExecutorService service = Executors.newSingleThreadScheduledExecutor();
        // 任务体，首次执行的延时时间，任务执行的间隔，间隔时间单位
        service.scheduleAtFixedRate(() -> insertQueue(null), 0, PERIOD_MILL_SECONDS, TimeUnit.MILLISECONDS);
    }

    private String getThrowableStr(Throwable throwable) {
        if (throwable == null) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        boolean isFirst = true;
        for (String s : Throwables.toStringList(throwable)) {
            if (isFirst) {
                isFirst = false;
            } else {
                sb.append(System.getProperty("line.separator"));
            }
            sb.append(s);
        }
        return sb.toString();
    }
    @PluginFactory
    public static DwLog4j2Appender createAppender(
            @PluginAttribute("name") String name,
            @PluginElement("Filter") final Filter filter,
            @PluginElement("Layout") Layout<? extends Serializable> layout,
            @PluginAttribute("endpoint") String endpoint,
            @PluginAttribute("app") String app) {
        if (name == null) {
            LOGGER.error("no name defined in conf.");
            return null;
        }
        if (layout == null) {
            layout = PatternLayout.createDefaultLayout();
        }
        return new DwLog4j2Appender(name, filter, (Layout) layout, endpoint, app);
    }

    public String getEndpoint() {
        return this.endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getApp() {
        return this.app;
    }

    public void setApp(String app) {
        this.app = app;
    }

    public CloseableHttpClient getHttpClient() {
        return this.client;
    }

    public void setHttpClient(CloseableHttpClient client) {
        this.client = client;
    }
}
