package com.digiwin.dmc.sdk.service.download;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.NetworkOptions;
import com.digiwin.dmc.sdk.config.SDKConstants;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.service.impl.UserManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


/**
 * @author chenzwd
 * @date : 2018-06-30 16:03
 * @Description:
 */
public class SegmentSegmentDocumentDownloader implements ISegmentDocumentDownloader {

    private String fileId;
    private int bufferSize;
    private String tenantId;
    private String bucketName;
    private String driveToken;
    private DownloadProcessState state;
    private FileOutputStream fileOutputStream;
    private IDownLoadCallbackable completeCallback;
    private IDownLoadCallbackable progressChangedCallback;
    private DownloadProgressEventArgs eventArgs = new DownloadProgressEventArgs();

    private static ExecutorService es = Executors.newCachedThreadPool();
    private static IUserManagerService userManagerService = UserManagerService.userInstance();

    public SegmentSegmentDocumentDownloader(String tenantId, String bucketName, String fileId, String saveToPath, int fileSize, String driveToken) {
        try {
            File file = new File(saveToPath);
            if (!file.exists()){
                file.createNewFile();
            }
            this.fileOutputStream = new FileOutputStream(file);
        } catch (Exception e) {
            throw  new OperateException(e);
        }
        this.fileId = fileId;
        this.bucketName = bucketName;
        this.driveToken = driveToken;
        this.eventArgs.setPercentage(0);
        this.eventArgs.setCompletedBytes(0);
        this.eventArgs.setTotalBytes(fileSize);
        this.tenantId = tenantId;
        this.state = DownloadProcessState.NotStarted;
        this.bufferSize = NetworkOptions.Default.getPackageSize();
    }


    /**
     * 异步下载文件内容
     */
    @Override
    public ISegmentDocumentDownloader beginDownload() {
        if (state != DownloadProcessState.NotStarted) {
            throw new OperateException("下载已开始，无法再次启动下载，请开启新的下载");
        }
        this.continueDownload();
        Runnable runnable= () -> multipartDownload(bucketName);
        es.execute(runnable);
        return this;
    }

    public void multipartDownload(String bucketName) {
        try {
            while (true) {
                int totalBytes = eventArgs.getTotalBytes();
                int bytesReaded = eventArgs.getCompletedBytes();
                if (state == DownloadProcessState.Stopped) {
                    return;
                } else if (state == DownloadProcessState.Paused) {
                    Thread.sleep(1000 * 60 * 60 * 24);
                    if (state == DownloadProcessState.Paused) {
                        state = DownloadProcessState.Timeout;
                        return;
                    }
                } else {
                    //from的位置是二进制数组的下标，所有需要把长度-1
                    int from = bytesReaded;
                    if (bytesReaded + bufferSize >= totalBytes) {
                        this.download(bucketName,from, totalBytes - bytesReaded);
                        eventArgs.setCompletedBytes(totalBytes);
                        eventArgs.setPercentage(1);
                        if (progressChangedCallback != null){
                            progressChangedCallback.callback(eventArgs);
                        }
                        if (completeCallback != null){
                            completeCallback.callback(eventArgs);
                        }
                        return;
                    } else {
                        this.download(bucketName,from, bufferSize);
                        eventArgs.setCompletedBytes(bytesReaded + bufferSize);
                        eventArgs.setPercentage((double) eventArgs.getCompletedBytes() / (double) totalBytes);
                        if (progressChangedCallback != null){
                            progressChangedCallback.callback(eventArgs);
                        }
                    }
                }
            }
        }catch (Exception e){
            throw new OperateException(e);
        }
    }

    public void download(String bucketName,int bytesReaded,int count){
        if (StringUtil.isEmpty(bucketName)){
            bucketName=ServerSetting.getBucketName();
        }
        String downloadResourceUrl = DmcUrl.getFilesResourceUrl() + bucketName +
                "/files/" + fileId+"/"+bytesReaded+"/"+count;
        try {
            Map<String, String> headers = new HashMap<>();
            if (StringUtil.isEmpty(driveToken)){
                String userToken = userManagerService.getUserToken(ServerSetting.getUser());
                headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
            }else{
                headers.put(SDKConstants.HttpHeaderDriveTokenKey, driveToken);
            }
            if (tenantId != null){
                headers.put(SDKConstants.HTTP_HEADER_TENANTID_KEY, tenantId);
            }
            this.fileOutputStream = HttpRequestUtil.downloadSegFile(downloadResourceUrl, headers,count,fileOutputStream);
        }catch (Exception e){
            throw new OperateException(e);
        }
    }

    /**
     *  设置下载进度
     * @param callbackable
     * @return
     */
    @Override
    public ISegmentDocumentDownloader onProgressChanged(IDownLoadCallbackable callbackable) {
        this.progressChangedCallback=callbackable;
        return this;
    }

    @Override
    public ISegmentDocumentDownloader onCompleted(IDownLoadCallbackable callbackable) {
        this.completeCallback=callbackable;
        return this;
    }

    /**
     * 暂停下载
     */
    @Override
    public void pauseDownload() {
        state = DownloadProcessState.Paused;
    }

    /**
     * 继续下载
     */
    @Override
    public void continueDownload() {
        state = DownloadProcessState.Downloading;
    }

    /**
     * 停止下载
     */
    @Override
    public void stopDownload() {
        state = DownloadProcessState.Stopped;
    }

}
