package com.digiwin.dmc.sdk.service.impl;

import com.digiwin.dmc.sdk.config.DmcUrl;
import com.digiwin.dmc.sdk.config.SDKConstants;
import com.digiwin.dmc.sdk.config.ServerSetting;
import com.digiwin.dmc.sdk.entity.UserInfo;
import com.digiwin.dmc.sdk.exception.OperateException;
import com.digiwin.dmc.sdk.identity.UserIdentity;
import com.digiwin.dmc.sdk.service.IUserManagerService;
import com.digiwin.dmc.sdk.util.HttpRequestUtil;
import com.digiwin.dmc.sdk.util.JwtUtil;
import com.digiwin.dmc.sdk.util.ObjectMapperUtil;
import com.digiwin.dmc.sdk.util.StringUtil;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class UserManagerService implements IUserManagerService {

    private static UserManagerService userManagerService;

    /**
     * 单一实例
     *
     * @return
     */
    public static IUserManagerService userInstance() {
        if (userManagerService == null) {
            userManagerService = new UserManagerService();
        }
        return userManagerService;
    }

    /**
     * 创建用户
     *
     * @param userInfo 用户信息
     * @return String
     */
    @Override
    public String createUser(UserInfo userInfo) {
        String identityUrl = DmcUrl.getUserResourceUrl();
        String userToken = this.getUserToken(ServerSetting.getAdmin());
        if (StringUtil.checkSpecialId(userInfo.getName())){
            throw new IllegalArgumentException("userName为空或包含特殊字符.号！");
        }
        Map<String, String> userObj = new HashMap<>();
        userObj.put("name", userInfo.getName());
        userObj.put("pwdHash", userInfo.getPassword());
        userObj.put("displayName", userInfo.getDisplayName());
        userObj.put("builtinRole", userInfo.getRole().toString());
        try {
            Map<String, String> headers = new HashMap<>();
            headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
            HashMap responseEntity = HttpRequestUtil.postJson(identityUrl,
                    ObjectMapperUtil.writeValueAsString(userObj), headers, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("user-id")) {
                return "";
            }
            return responseEntity.get("user-id").toString();
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }

    /**
     * 查询用户
     *
     * @param userName 用户名
     * @return 类
     */
    @Override
    public UserInfo getUser(String userName) {
        if (userName == null || userName.length() == 0) {
            throw new IllegalArgumentException("参数userName is null");
        }
        String userToken = getUserToken(ServerSetting.getAdmin());
        String url = DmcUrl.getUserResourceUrl();
        Map<String, String> headers = new HashMap<>();
        headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        UserInfo userInfo = HttpRequestUtil.get(url + "/" + userName, headers, UserInfo.class);
        return userInfo;
    }

    /**
     * 查询
     *
     * @return list
     */
    @Override
    public List<UserInfo> getUsers() {
        String url = DmcUrl.getUserResourceUrl();
        String userToken = getUserToken(ServerSetting.getAdmin());
        Map<String, String> headers = new HashMap<>();
        headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        ArrayList users = HttpRequestUtil.get(url, headers, ArrayList.class);
        List<UserInfo> userInfos=new ArrayList<>();
        if (users!=null){
            for (Object user : users) {
                try {
                    String json = ObjectMapperUtil.writeValueAsString(user);
                    UserInfo userInfo = ObjectMapperUtil.readValue(json, UserInfo.class);
                    userInfos.add(userInfo);
                } catch (IOException e) {
                    throw new OperateException(e);
                }
            }
        }
        return userInfos;
    }

    /**
     * 判断用户是否存在
     *
     * @param userName 用户名
     * @return boolean
     */
    @Override
    public String isExistUser(String userName) {
        if (userName == null) {
            throw new IllegalArgumentException("参数userIdentity is null");
        }
        String url = DmcUrl.getUserResourceUrl() + "/" + userName + "/isExist";
        String userToken = getUserToken(ServerSetting.getAdmin());
        Map<String, String> headers = new HashMap<>();
        headers.put(SDKConstants.HttpHeaderUserTokenKey, userToken);
        HashMap isExist=HttpRequestUtil.get(url, headers, HashMap.class);
        if (isExist==null||!isExist.containsKey("isExist")){
            return null;
        }
        return isExist.get("isExist").toString();
    }

    HashMap<Integer,String> cache=new HashMap<Integer,String>();

    /**
     * 获取用户token
     *
     * @param userIdentity 用户身份
     * @return String
     */
    @Override
    public String getUserToken(UserIdentity userIdentity) {
        if (userIdentity == null) {
            throw new IllegalArgumentException("参数userIdentity is null");
        }
        String userToken;
        int hashcode=  userIdentity.hashCode();
        if (cache.containsKey(hashcode)){
            userToken = cache.get(hashcode);
            if (!JwtUtil.isExpires(userToken)){
                return  userToken;
            }
        }

        String identityUrl = DmcUrl.getIdentityResourceUrl() + "/" + "login";
        Map<String, String> parameters = new HashMap<>();
        parameters.put("username", userIdentity.getUserName());
        parameters.put("pwdhash", userIdentity.getPasswordHash());
        try {
            String json=ObjectMapperUtil.writeValueAsString(parameters);
            HashMap responseEntity = HttpRequestUtil.postJson(identityUrl, json, HashMap.class);
            if (responseEntity == null || !responseEntity.containsKey("userToken")) {
                cache.remove(hashcode);
                return null;
            }
            userToken= responseEntity.get("userToken").toString();
            cache.put(hashcode,userToken);
            return  userToken;
        } catch (Exception e) {
            throw new OperateException(e);
        }
    }
}
