package com.digiwin.mobile.mobileuibot.agileData.sse.listener;

import com.digiwin.mobile.mobileuibot.agileData.sse.domain.EventData;
import com.digiwin.mobile.mobileuibot.agileData.sse.domain.SseAniaEventEnum;
import com.digiwin.mobile.mobileuibot.agileData.sse.dto.SSEBaseEvent;
import com.digiwin.mobile.mobileuibot.agileData.sse.service.impl.event.SSEOnEventService;
import com.digiwin.mobile.mobileuibot.agileData.sse.utils.SseEmitterUtils;
import com.digiwin.mobile.mobileuibot.agileData.sse.utils.SseEventUtils;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.locale.service.impl.LocaleServiceImpl;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Response;
import okhttp3.sse.EventSource;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.util.Objects;

/**
 * ADT事件监听
 */
@Slf4j
public class SSEAdtListener extends SSEBaseListener {

    private final SSEBaseEvent sseBaseEvent;


    public SSEAdtListener(SSEBaseEvent sseBaseEvent) {
        this.sseBaseEvent = sseBaseEvent;
    }

    /**
     * 通道连接成功事件
     * @param eventSource event
     * @param response res
     */
    @Override
    public void onOpen(EventSource eventSource, Response response) {
        sseBaseEvent.setScrumEventSource(eventSource);
    }

    /**
     * 监听adt推送事件
     * @param eventSource event
     * @param id id
     * @param type type
     * @param data data
     */
    @Override
    public void onEvent(EventSource eventSource, @Nullable String id, @Nullable String type, String data) {
        if (SseAniaEventEnum.CHAT_CREATED.getEvent().equals(type)) {
            return;
        }
        SSEOnEventService sseOnEventService = SpringContextHolder.getBean(SSEOnEventService.class);
        sseOnEventService.onEvent(sseBaseEvent, eventSource, type, data);

    }

    /**
     * 管道关闭事件
     * @param eventSource event
     */
    @Override
    public void onClosed(EventSource eventSource) {
        //标记ADT对话结束
        sseBaseEvent.setIsScrumOver(true);
        //ADT是否推送存在结束信息
        if (!SseEventUtils.noScrumTerminate(sseBaseEvent.getScrumEventDatas())) {
            SseEmitterUtils.sendChatErrorComplete(sseBaseEvent.getAniaEmitter());
            SseEmitterUtils.sendAnswerDone(sseBaseEvent.getAniaEmitter(), false);
            sseBaseEvent.getAniaEmitter().complete();
        }

    }

    /**
     * 监听通道异常事件
     * @param eventSource event
     * @param t t
     * @param response res
     */
    @Override
    public void onFailure(EventSource eventSource, @Nullable Throwable t, @Nullable Response response) {
        if (t != null) {
            log.info("ADT监听出现异常，异常信息:{}", t.getMessage());
        }
        sseBaseEvent.setIsScrumOver(true);

        LocaleService localeService = SpringContextHolder.getBean(LocaleServiceImpl.class);
        String errorMsg = localeService.getLanguageValue(sseBaseEvent.getLang(), "大模型访问异常，请向客服人员反馈或稍后重试。");
        String answer;
        if (Objects.nonNull(response) && response.isSuccessful()) {
            if (t instanceof IllegalStateException && t.getMessage().startsWith("ResponseBodyEmitter has already completed")) {
                return;
            }
        }
        answer = errorMsg;
        if (StringUtils.isNotBlank(answer)) {
            EventData eventData = new EventData(SseAniaEventEnum.MESSAGE_DELTA.getEvent());
            eventData.setData(new EventData.EventMessage(SseAniaEventEnum.SseEventDataTypeEnum.ANSWER.getType(), SseAniaEventEnum.EventDataMessageEnum.TEXT.getMessageType(), answer));
            SseEmitterUtils.send(sseBaseEvent.getAniaEmitter(), eventData);
            SseEmitterUtils.sendChatErrorComplete(sseBaseEvent.getAniaEmitter());
            SseEmitterUtils.sendAnswerDone(sseBaseEvent.getAniaEmitter(), false);
            sseBaseEvent.getAniaEmitter().complete();
        }
    }

}