package com.digiwin.mobile.mobileuibot.agileData.sse.service.impl;

import com.digiwin.mobile.mobileuibot.agileData.sse.api.SSEManagerService;
import com.digiwin.mobile.mobileuibot.agileData.sse.domain.AthenaMessageEvent;
import com.digiwin.mobile.mobileuibot.agileData.sse.domain.EventData;
import com.digiwin.mobile.mobileuibot.agileData.sse.domain.SseAniaEventEnum;
import com.digiwin.mobile.mobileuibot.agileData.sse.dto.AgileDadaSseChatReqDTO;
import com.digiwin.mobile.mobileuibot.agileData.sse.dto.AthenaMessageDTO;
import com.digiwin.mobile.mobileuibot.agileData.sse.dto.SSEBaseEvent;
import com.digiwin.mobile.mobileuibot.agileData.sse.listener.SSEAdtListener;
import com.digiwin.mobile.mobileuibot.agileData.sse.service.SseService;
import com.digiwin.mobile.mobileuibot.agileData.sse.service.impl.process.SSEProcessService;
import com.digiwin.mobile.mobileuibot.agileData.sse.utils.SseEmitterUtils;
import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.log.TraceIdUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

/**
 * @date 2025/9/2
 * @author yanfeng
 */
@Slf4j
@Service
public class SseServiceImpl implements SseService {

    @Resource(name = "defaultThreadPool")
    private ThreadPoolTaskExecutor threadPoolTaskExecutor;

    @Resource
    private SSEManagerService sseManagerService;

    @Resource
    private SSEProcessService sseProcessService;

    @Override
    public SseEmitter sseChat(AgileDadaSseChatReqDTO queryDTO) {
        log.info("sseChat入参:\n{}", JsonUtil.javaObjectToJsonString(queryDTO));
        SSEBaseEvent sseBaseEvent = new SSEBaseEvent();
        SseEmitter sseEmitter = sseManagerService.generateSseEmitter(sseBaseEvent);
        sseBaseEvent.setAniaEmitter(sseEmitter);

        AthenaMessageEvent event = handleEvent(queryDTO);
        event.setSseMessage(true);
        event.setSseEmitter(sseEmitter);
        event.setAppCode(queryDTO.getAssistantId());

        Map<String, Object> map = JsonUtil.objectToJavaObject(queryDTO, new TypeReference<Map<String, Object>>() {
        });
        CompletableFuture.runAsync(() -> {
            // 1.1 异步开启会话
            EventData eventData = new EventData(SseAniaEventEnum.CHAT_CREATED.getEvent(), null);
            if (Objects.nonNull(sseEmitter)) {
                SseEmitterUtils.send(sseEmitter, eventData);
            }
            SSEAdtListener sseListener = new SSEAdtListener(sseBaseEvent);
            // 2.4 初始化sse , ania sse , 创建adt sse 连接
            SSEBaseEvent.builderSchemaEvent(event, sseBaseEvent);
            sseProcessService.process(sseListener, event, map, sseBaseEvent);
        }, threadPoolTaskExecutor);

        return sseEmitter;
    }

    /**
     * 处理入参
     * @param req
     * @return
     */
    public AthenaMessageEvent handleEvent(AgileDadaSseChatReqDTO req) {
        AthenaMessageEvent event = new AthenaMessageEvent();
        Map<String, Object> athenaMessage = new HashMap<>();

        // 使用Optional处理可能为null的req.getMessage()和req.getMessage().getExtData()
        Map<String, Object> extData = Optional.ofNullable(req.getMessage())
                .map(AgileDadaSseChatReqDTO.Message::getExtData)
                .orElse(new HashMap<>());

        athenaMessage.put("msgExt", extData);
        Map<String, Object> msgBody = new HashMap<>();
        msgBody.put("text", extData.get("msg"));
        athenaMessage.put("msgBody", msgBody);
        AthenaMessageDTO athenaMessageDTO = JsonUtil.objectToJavaObject(athenaMessage, AthenaMessageDTO.class);

        // 使用Optional处理可能为null的消息属性
        Optional.ofNullable(req.getMessage())
                .ifPresent(message -> {
                    athenaMessageDTO.setMessageId(message.getMessageId());
                });

        athenaMessageDTO.setConversationId(req.getConversationId());
        athenaMessageDTO.setAppCode(req.getAssistantId());
        // 初始化上下文业务信息
        this.initEventInfo(athenaMessageDTO, event);
        //初始化助理编码
        this.analyzeAsaCode(athenaMessageDTO, event);
        event.setUserId(req.getUserId());
        event.setUserName(req.getUserName());
        event.setTenantId(req.getTenantId());
        event.setTenantName(req.getTenantName());
        event.setToken(req.getIamUserToken());
        event.setLang(req.getLocale());
        return event;
    }


    /**
     * 根据不通场景解析asaCode助理编码
     * 优先级 asaCode > assistantCode > athenaMessageDTO.getAsaCode();
     * @param athenaMessageDTO 入参
     * @param event 上下文
     */
    public void analyzeAsaCode(AthenaMessageDTO athenaMessageDTO, AthenaMessageEvent event) {
        if (Objects.isNull(athenaMessageDTO.getMsgExt())) {
            event.setAsaCode(athenaMessageDTO.getAsaCode());
            return;
        }
        String asaCode = Optional.ofNullable(athenaMessageDTO.getMsgExt().get("asaCode"))
                .map(Object::toString)
                .filter(StringUtils::isNotEmpty)
                .orElseGet(() -> Optional.ofNullable(athenaMessageDTO.getMsgExt().get("assistantCode"))
                        .map(Object::toString)
                        .filter(StringUtils::isNotEmpty)
                        .orElse(athenaMessageDTO.getAsaCode()));
        event.setAsaCode(asaCode);
    }

    /**
     * 初始化上下文业务信息字段
     * @param athenaMessageDTO 入参
     * @param event 上下文
     */
    private void initEventInfo(AthenaMessageDTO athenaMessageDTO, AthenaMessageEvent event) {

        // 相关推荐/猜你想问默认1
        event.setAnswerResult(1);
        // 默认发送娜娜
        event.setSendNana(true);
        // 链路id
        String ptxId = TraceIdUtil.get();
        event.setPtxId(ptxId);
        event.setMsgBody(athenaMessageDTO.getMsgBody());
        event.setMsgExt(athenaMessageDTO.getMsgExt());
        event.setAskTime(LocalDateTime.now());
        // 唯一序列顶替actionId
        Long generateSerialNo = UUIDUtil.getLongUuid();
        String question = MapUtils.getString(event.getMsgBody(), "text");
        event.setQuestion(question);
        // 技能标识
        String skillType = MapUtils.getString(athenaMessageDTO.getMsgExt(), "agileDataSkillType");
        event.setSkillType(skillType);
        event.setGenerateSerialNo(generateSerialNo);
        log.info("agileData_mobile_generateSerialNo:{}", generateSerialNo);
    }

}