package com.digiwin.mobile.mobileuibot.api;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>功能描述：枚举约束验证器</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: EnumConstraintValidator.java
 * @Author: wangjwc
 * @Date: created at 2025/8/6 15:54
 */
public class EnumConstraintValidator implements ConstraintValidator<EnumConstraint, Object> {

    // 枚举中调用valueMethod方法得到的值
    private List<Object> values = new ArrayList<>();
    private Class<? extends Enum<?>> enumClass;

    @Override
    public void initialize(EnumConstraint constraintAnnotation) {
        enumClass = constraintAnnotation.enumClass();
        // 值方法名称
        String valueMethod = constraintAnnotation.valueMethod();

        try {
            // 反射获取枚举值
            Method method = enumClass.getDeclaredMethod(valueMethod);
            values = Arrays.stream(enumClass.getEnumConstants())
                    .map(e -> {
                        try {
                            return method.invoke(e);
                        } catch (Exception ex) {
                            throw new RuntimeException(ex);
                        }
                    })
                    .collect(Collectors.toList());
        } catch (NoSuchMethodException e) {
            throw new RuntimeException("枚举类" + enumClass.getName() + "缺少方法: " + valueMethod, e);
        }
    }

    @Override
    public boolean isValid(Object value, ConstraintValidatorContext context) {
        // 允许为空，结合@NotNull、@NotBlank注解一起使用
        if (value == null || value.toString().trim().isEmpty()) {
            return true;
        }
        // 检查值是否在枚举中
        if (values.contains(value)) {
            return true;
        }
        // 动态替换错误消息中的占位符
        String resolvedMessage = context.getDefaultConstraintMessageTemplate()
                .replace("{value}", value.toString())
                .replace("{enumClass}", enumClass.getSimpleName())
                .replace("{values}", values.stream().map(Object::toString).collect(Collectors.joining(", ")));
        // 禁用默认消息，使用自定义消息
        context.disableDefaultConstraintViolation();
        context.buildConstraintViolationWithTemplate(resolvedMessage)
                .addConstraintViolation();

        return false;
    }
}