package com.digiwin.mobile.mobileuibot.cache;

import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentConfig;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentConfigService;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.RemovalCause;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 功能: 缓存内部地址工具类
 *
 * @author: wjw
 * @date: created at 2023/1/28
 */
@Component
public class JvmCache {

    private static final Logger log = LoggerFactory.getLogger(JvmCache.class);

    /**
     * 智驱中台 是否强校验租户ID必传开关配置
     */
    private static final String ATHENA_API_SWITCH = "athena_api_switch_config";

    @Autowired
    private EnvironmentConfigService environmentConfigService;

    /**
     * 使用LoadingCache本地缓存
     * 定时刷新，自动存储，主要是内部的重写
     * 相当于Map
     * 关注下maxSize的数量,根据业务量进行评估
     */
    private LoadingCache<String, Boolean> athenaApiSwitchCache = Caffeine.newBuilder()
            .recordStats()
            // 初始的缓存空间大小
            .initialCapacity(8)
            // 缓存的最大条数
            .maximumSize(8)
            // 设置最后一次写入后经过固定时间会被刷新（前提是有被访问过）
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            // 移除监听
            .removalListener((String key, Boolean value, RemovalCause cause)
                    -> log.debug("JvmCache[cahceBuilder] key {} msg {} ", key, cause))
            .build(new CacheLoader<String, Boolean>() {
                @Override
                public Boolean load(@NonNull String key) throws Exception {
                    return generateValueByKey(key);
                }

                @Override
                public Boolean reload(@NonNull String key, @NonNull Boolean oldValue) throws Exception {
                    log.debug("reload cache activated...key is {}, oldValue is {}", key, JsonUtil.javaObjectToJsonString(oldValue));
                    return this.load(key);
                }
            });

    private Boolean generateValueByKey(String key) {
        Boolean athenaApiSwitch = Boolean.FALSE;
        List<EnvironmentConfig> environmentConfig = environmentConfigService.getEnvironmentConfig();
        try {
            athenaApiSwitch = environmentConfig.get(0).getConfig().getServerConfig().getAthenaApiSwitch();
        } catch (Exception ignored) {
        }
        return athenaApiSwitch;
    }

    /**
     * 获取 是否强校验租户ID必传开关
     */
    public Boolean getAthenaApiSwitch() {
        return getValue(ATHENA_API_SWITCH);
    }

    /**
     * 根据key从缓存中取值
     *
     * @param key
     * @return
     */
    public Boolean getValue(String key) {
        try {
            return athenaApiSwitchCache.get(key);
        } catch (Exception e) {
            log.error("JvmCache[getValue] occurred error ", e);
            return false;
        }
    }
}
