package com.digiwin.mobile.mobileuibot.cache;

import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentConfig;
import com.digiwin.mobile.mobileuibot.environment.EnvironmentConfigService;
import com.digiwin.mobile.mobileuibot.locale.service.TranslateService;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.RemovalCause;
import org.apache.commons.lang3.StringUtils;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 功能: LuckyCola缓存类
 *
 * @author: xujinc
 * @date: created at 2023/10/10
 */

@Component
public class LuckyColaCache {

    private static final Logger log = LoggerFactory.getLogger(LuckyColaCache.class);

    /**
     * 智驱中台 是否强校验租户ID必传开关配置
     */
    private static final String LUCKYCOLA_COLAKEY = "LuckyCola_ColaKey";

    @Autowired
    private TranslateService translateService;

    /**
     * 使用LoadingCache本地缓存
     * 定时刷新，自动存储，主要是内部的重写
     * 相当于Map
     * 关注下maxSize的数量,根据业务量进行评估
     */
    private LoadingCache<String, String> colaKeyCache = Caffeine.newBuilder()
            .recordStats()
            // 初始的缓存空间大小
            .initialCapacity(8)
            // 缓存的最大条数
            .maximumSize(8)
            // 设置最后一次写入后经过固定时间会被刷新（ColaKey刷新间隔时间是60天）
            .refreshAfterWrite(60, TimeUnit.DAYS)
            // 移除监听
            .removalListener((String key, String value, RemovalCause cause)
                    -> log.debug("LuckyColaCache[cahceBuilder] key {} msg {} ", key, cause))
            .build(new CacheLoader<String, String>() {
                @Override
                public String load(@NonNull String key) throws Exception {
                    return translateService.getColaKey();
                }

                @Override
                public String reload(@NonNull String key, @NonNull String oldValue) throws Exception {
                    log.debug("luckyCola reload cache activated...key is {}, oldValue is {}", key, JsonUtil.javaObjectToJsonString(oldValue));
                    return this.load(key);
                }
            });

    /**
     * 获取 ColaKey
     */
    public String getColaKey() {
        return getValue(LUCKYCOLA_COLAKEY);
    }

    /**
     * 根据key从缓存中取值
     *
     * @param key
     * @return
     */
    public String getValue(String key) {
        try {
            return colaKeyCache.get(key);
        } catch (Exception e) {
            log.error("LuckyColaCache[getValue] occurred error ", e);
            return StringUtils.EMPTY;
        }
    }

    /**
     * 根据key从缓存中取值
     *
     * @param value
     * @return
     */
    public void putValue(String value) {
        try {
            colaKeyCache.put(LUCKYCOLA_COLAKEY, value);
        } catch (Exception e) {
            log.error("LuckyColaCache[putValue] occurred error ", e);
        }
    }
}
