package com.digiwin.mobile.mobileuibot.common.exceptionhandle;

import com.digiwin.mobile.mobileuibot.api.ApiResponse;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.context.request.async.AsyncRequestTimeoutException;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;
import java.util.stream.Collectors;

/**
 * <p>功能描述：全局统一异常处理类</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: MobileUiBotExceptionHandler
 * @Author: Zaregoto
 * @Date: 2022/2/7 21:10
 */
@ControllerAdvice
public class MobileUiBotExceptionHandler {

    private static final String TOKEN_EXPIRED_CN = "登录过期";
    private static final String TOKEN_EXPIRED_TW = "登錄過期";
    private static final String TOKEN_EXPIRED_US = "Login expired";

    private static final String RENDER_PAGE_URI = "/mobile/v1/uibot/model";

    @Autowired
    private LocaleService localeService;

    @ExceptionHandler(value = Exception.class)
    @ResponseBody
    public ApiResponse exceptionHandler(Exception e) {
        e.printStackTrace();
        if (null != e.getMessage()
                && (e.getMessage().contains(TOKEN_EXPIRED_CN)
                || e.getMessage().contains(TOKEN_EXPIRED_TW)
                || e.getMessage().contains(TOKEN_EXPIRED_US))) {
            return buildNoAuth(e);
        } else {
            return ApiResponse.buildError(e.getMessage(), ExceptionUtils.getStackTrace(e));
        }
    }

    @ExceptionHandler(value = ServiceException.class)
    @ResponseBody
    public ApiResponse serviceExceptionHandler(Exception e, HttpServletRequest request) {
        e.printStackTrace();
        if (null != e.getMessage()
                && (e.getMessage().contains(TOKEN_EXPIRED_CN)
                || e.getMessage().contains(TOKEN_EXPIRED_TW)
                || e.getMessage().contains(TOKEN_EXPIRED_US))) {
            return buildNoAuth(e);
        }
        return ApiResponse.buildServiceError(e.getMessage(), ExceptionUtils.getStackTrace(e));
    }

    public ApiResponse buildNoAuth(Exception e) {
        String message;
        if (e.getMessage().contains(":")) {
            message = e.getMessage().split(":")[1];
        } else {
            message = e.getMessage();
        }
        ApiResponse response = ApiResponse.buildError(message);
        response.setCode("401");
        return response;
    }

    /**
     * 处理请求参数格式错误 @RequestBody上validate失败后抛出的异常是MethodArgumentNotValidException异常。
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = {BindException.class, ValidationException.class, MethodArgumentNotValidException.class})
    @ResponseBody
    public ApiResponse<?> validatedExceptionHandler(Exception e) {
        if (e instanceof MethodArgumentNotValidException) {
            MethodArgumentNotValidException ex = (MethodArgumentNotValidException) e;
            return ApiResponse.buildError(ex.getBindingResult().getAllErrors().stream()
                    .map(error -> localeService.getLanguageValue(AppRequestContext.getContextEntity().getLocale(), error.getDefaultMessage()))
                    .collect(Collectors.joining("; ")));
        } else if (e instanceof BindException) {
            BindException ex = (BindException) e;
            return ApiResponse.buildError(ex.getAllErrors().stream()
                    .map(error -> localeService.getLanguageValue(AppRequestContext.getContextEntity().getLocale(), error.getDefaultMessage()))
                    .collect(Collectors.joining("; ")));
        } else if (e instanceof ConstraintViolationException) {
            ConstraintViolationException ex = (ConstraintViolationException) e;
            return ApiResponse.buildError(ex.getConstraintViolations().stream()
                    .map(error -> localeService.getLanguageValue(AppRequestContext.getContextEntity().getLocale(), error.getMessage()))
                    .collect(Collectors.joining("; ")));
        } else {
            return ApiResponse.buildError(e.getMessage());
        }
    }

    /**
     * 系统异步调用异常捕获
     *
     * @param ex
     * @return
     */
    @ExceptionHandler(AsyncRequestTimeoutException.class)
    @ResponseStatus(HttpStatus.REQUEST_TIMEOUT) // 返回 408 状态码
    public ApiResponse handleAsyncRequestTimeoutException(AsyncRequestTimeoutException ex) {
        return ApiResponse.buildError().setData("The request timed out. Please try again later");
    }
}
