package com.digiwin.mobile.mobileuibot.common.http;

import com.digiwin.mobile.mobileuibot.common.calculate.UUIDUtil;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.log.TraceIdUtil;
import com.digiwin.mobile.mobileuibot.common.logrecord.OuterRequestRecordManage;
import com.digiwin.mobile.mobileuibot.common.logrecord.RequestSourceEnum;
import com.digiwin.mobile.mobileuibot.common.string.StringUtil;
import com.digiwin.mobile.mobileuibot.model.db1.OuterRequestRecord;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpRequest;
import org.springframework.http.HttpStatus;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.util.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.util.*;

/**
 * 请求三方接口调用请求、响应详细信息记录
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: DigiwinHttpReqMsgInterceptor
 * @Author: Zaregoto
 * @Date: 2021/4/29 21:58
 */
public class DigiwinHttpReqMsgInterceptor implements ClientHttpRequestInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(DigiwinHttpReqMsgInterceptor.class);

    @Autowired
    private OuterRequestRecordManage outerRequestRecordManage;

    /**
     * 外部请求日志中记录的响应内容大小上限，相当于字节数（Byte）
     */
    private static final int RECORD_RESPONSE_BODY_STRING_MAX_SIZE = 64 * 1024;

    @Override
    public ClientHttpResponse intercept(HttpRequest httpRequest, byte[] bytes,
                                        ClientHttpRequestExecution clientHttpRequestExecution) throws IOException {
        long startTime = System.currentTimeMillis();
        ClientHttpResponse response = clientHttpRequestExecution.execute(httpRequest, bytes);
        long endTime = System.currentTimeMillis();

        // 参考链接：https://blog.csdn.net/hellopeng1/article/details/83662544
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        byte[] respBytes = new byte[512];
        int length;
        while ((length = response.getBody().read(respBytes)) > 0) {
            outputStream.write(respBytes, 0, length);
        }

        try {
            // 封装跟踪日志
            this.buildOuterRequestRecord(httpRequest, bytes, response, outputStream, startTime, endTime);
        } catch (Exception e) {
            logger.error("DigiwinHttpReqMsgInterceptor-拦截跟踪日志异常-intercept：", e);
        }

        if (logger.isInfoEnabled()) {
            traceReqAndResp(httpRequest, bytes, response, outputStream);
        }
        return new ClientHttpResponseWrapper(response, outputStream.toByteArray());
    }

    /**
     * 封装跟踪日志
     *
     * @param request
     * @param bytes
     * @param response
     * @param outputStream
     * @param startTime
     * @param endTime
     */
    private void buildOuterRequestRecord(HttpRequest request, byte[] bytes, ClientHttpResponse response,
                                         ByteArrayOutputStream outputStream, long startTime, long endTime)
            throws IOException {
        String outerRequestUrl = request.getURI().toString();
        /**
         * 平台ATDM的响应，不记录完整的响应日志；未来如果要扩展条件，加到如下布尔值判断即可
         * 因ATDM直接和应用API接通，会记录大量业务数据，这会快速增加mysql的binlog大小，进而影响平台SD侦测引擎的稳定性（数据量过大导致OOM）
         * add by mowj @2023.11.20
         */
        boolean needTruncateRespBody = outerRequestUrl.toLowerCase().contains("atdm");
        OuterRequestRecord.Request requestObj = this.buildRequest(request, bytes);
        OuterRequestRecord.Response responseObj = this.buildResponse(response, outputStream, needTruncateRespBody);

        OuterRequestRecord requestRecord = new OuterRequestRecord()
                .setRequestId(UUIDUtil.getUuid())
                .setTraceId(this.buildTraceId(response, outputStream))
                .setMethod(Optional.ofNullable(request.getMethod()).map(Enum::name).orElse(""))
                .setSource(RequestSourceEnum.OUTER.getSource())
                .setUrl(outerRequestUrl)
                .setRequestObj(requestObj)
                .setResponseObj(responseObj)
                .setRequestTime(new Timestamp(startTime))
                .setResponseTime(new Timestamp(endTime))
                .setCode(response.getStatusCode().value())
                .setMillis(endTime - startTime);

        // 队列存储
        outerRequestRecordManage.add(requestRecord);
    }

    private String buildTraceId(ClientHttpResponse response, ByteArrayOutputStream outputStream) {
        String traceId = TraceIdUtil.get();
        // 优先使用响应中的traceId
        try {
            String bodyStr = new String(outputStream.toByteArray(), StandardCharsets.UTF_8);
            if (StringUtils.hasLength(bodyStr)) {
                String s = StringUtil.valueOf(JsonUtil.jsonStringToObject(bodyStr, Map.class).get("traceId"));
                if (StringUtils.hasLength(s)) {
                    return s;
                }
            }
        } catch (Exception ignored) {
        }
        // 若响应中没有traceId，则使用请求头中的traceId
        HttpHeaders headers = response.getHeaders();
        try {
            List<String> strings = headers.get("ja-trace-id");
            if (CollectionUtils.isNotEmpty(strings)) {
                String jaTraceId = strings.get(0);
                if (StringUtils.hasLength(jaTraceId)) {
                    return jaTraceId;
                }
            }
        } catch (Exception ignored) {
        }
        try {
            List<String> strings = headers.get("traceId");
            if (CollectionUtils.isNotEmpty(strings)) {
                String s = strings.get(0);
                if (StringUtils.hasLength(s)) {
                    return s;
                }
            }
        } catch (Exception ignored) {
        }
        return traceId;
    }

    private OuterRequestRecord.Request buildRequest(HttpRequest request, byte[] reqBody) {
        String bodyStr = new String(reqBody, StandardCharsets.UTF_8);
        Object body;
        try {
            body = StringUtils.hasLength(bodyStr) ? JsonUtil.jsonStringToObject(bodyStr, Object.class) : new HashMap<>();
        } catch (Exception e) {
            logger.error("DigiwinHttpReqMsgInterceptor request parse error:", e);
            body = bodyStr;
        }
        OuterRequestRecord.Request reqRecord = new OuterRequestRecord.Request();
        reqRecord.setHeader(this.headersToMap(request.getHeaders()))
                .setQueryString(request.getURI().getRawQuery())
                .setBody(body);
        return reqRecord;
    }

    private OuterRequestRecord.Response buildResponse(ClientHttpResponse response, ByteArrayOutputStream outputStream,
                                                      boolean needTruncateBody) {
        String bodyStr = new String(outputStream.toByteArray(), StandardCharsets.UTF_8);
        Object body;
        try {
            if (StringUtils.hasLength(bodyStr)) {
                if (needTruncateBody && bodyStr.length() > RECORD_RESPONSE_BODY_STRING_MAX_SIZE) {
                    body = bodyStr.substring(0, RECORD_RESPONSE_BODY_STRING_MAX_SIZE).concat("……");
                } else {
                    body = JsonUtil.jsonStringToObject(bodyStr, Object.class);
                }
            } else {
                body = Collections.emptyMap();
            }
        } catch (Exception e) {
            logger.error("DigiwinHttpReqMsgInterceptor response parse error:", e);
            if (StringUtils.hasLength(bodyStr)) {
                if (needTruncateBody && bodyStr.length() > RECORD_RESPONSE_BODY_STRING_MAX_SIZE) {
                    body = bodyStr.substring(0, RECORD_RESPONSE_BODY_STRING_MAX_SIZE).concat("……");
                } else {
                    body = bodyStr;
                }
            } else {
                body = "";
            }
        }

        OuterRequestRecord.Response respRecord = new OuterRequestRecord.Response();
        respRecord.setHeader(this.headersToMap(response.getHeaders()))
                .setBody(body);
        return respRecord;
    }

    /**
     * 将http头信息格式化处理
     *
     * @param httpHeaders
     * @return
     */
    private Map<String, Object> headersToMap(HttpHeaders httpHeaders) {
        Map<String, Object> headers = new HashMap<>();
        if (null == httpHeaders) {
            return headers;
        }
        httpHeaders.forEach((key, values) -> headers.put(key, String.join(", ", values)));
        return headers;
    }

    /**
     * 记录请求和响应参数
     *
     * @param request
     * @param reqBody
     * @param response
     * @param respOutputStream
     * @throws IOException
     */
    private void traceReqAndResp(HttpRequest request, byte[] reqBody,
                                 ClientHttpResponse response, ByteArrayOutputStream respOutputStream) throws IOException {
        logger.info(
                "\n===========================外部HTTP请求============================\n"
                        + "URI         : {}\n"
                        + "Method      : {}\n"
                        + "Headers     : {}\n"
                        + "Request body: {}\n"
                        + "==========================外部HTTP响应============================\n"
                        + "Status code  : {}\n"
                        + "Status text  : {}\n"
                        + "Headers      : {}\n"
                        + "Response body: {}\n"
                        + "",
                request.getURI(),
                request.getMethod(),
                request.getHeaders(),
                new String(reqBody, StandardCharsets.UTF_8),
                response.getStatusCode(),
                response.getStatusText(),
                response.getHeaders(),
                new String(respOutputStream.toByteArray(), StandardCharsets.UTF_8)
        );
    }

    class ClientHttpResponseWrapper implements ClientHttpResponse {
        private ClientHttpResponse response;
        private InputStream in;

        public ClientHttpResponseWrapper(ClientHttpResponse response, byte[] body) {
            this.response = response;
            this.in = new ByteArrayInputStream(body);
        }

        @Override
        public HttpStatus getStatusCode() throws IOException {
            return this.response.getStatusCode();
        }

        @Override
        public int getRawStatusCode() throws IOException {
            return this.response.getRawStatusCode();
        }

        @Override
        public String getStatusText() throws IOException {
            return this.response.getStatusText();
        }

        @Override
        public void close() {
            try {
                this.in.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        @Override
        public InputStream getBody() throws IOException {
            return in;
        }

        @Override
        public HttpHeaders getHeaders() {
            return response.getHeaders();
        }
    }
}
