package com.digiwin.mobile.mobileuibot.common.http.impl;

import com.digiwin.mobile.mobileuibot.cache.JvmCache;
import com.digiwin.mobile.mobileuibot.common.context.AppRequestContext;
import com.digiwin.mobile.mobileuibot.common.exception.ServiceException;
import com.digiwin.mobile.mobileuibot.common.http.HttpService;
import com.digiwin.mobile.mobileuibot.common.http.model.AthenaBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.common.json.JsonUtil;
import com.digiwin.mobile.mobileuibot.common.localization.LocaleUtil;
import com.digiwin.mobile.mobileuibot.common.log.TraceIdUtil;
import com.digiwin.mobile.mobileuibot.config.SysEnvConfig;
import com.digiwin.mobile.mobileuibot.proxy.digiwhale.model.DigiwhaleBasicHttpRequestHead;
import com.digiwin.mobile.mobileuibot.proxy.enums.AthenaCheckApiEnum;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.SneakyThrows;
import org.apache.commons.lang3.BooleanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartFile;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * <p>功能描述：Http请求服务</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: HttpServiceImpl
 * @Author: Zaregoto
 * @Date: 2021/4/21 2:18
 */
@Service("httpService")
public class HttpServiceImpl implements HttpService {
    private static final Logger logger = LoggerFactory.getLogger(HttpServiceImpl.class);

    @Autowired
    private SysEnvConfig sysEnvConfig;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private JvmCache jvmCache;

    /**
     * 构建 外部接口调用 请求头参数
     *
     * @param url                        外部Url地址
     * @param athenaBasicHttpRequestHead 鼎捷Athena系统API请求的基础头部数据模型
     * @return
     */
    private HttpHeaders processHeaders(String url, AthenaBasicHttpRequestHead athenaBasicHttpRequestHead) {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.set("digi-middleware-auth-app", sysEnvConfig.getAppToken());
        httpHeaders.set("digi-middleware-auth-user", athenaBasicHttpRequestHead.getUserToken());
        httpHeaders.set("token", athenaBasicHttpRequestHead.getUserToken());
        httpHeaders.set("client-agent", athenaBasicHttpRequestHead.getClientAgent());
        httpHeaders.set("locale", athenaBasicHttpRequestHead.getLocale());
        httpHeaders.set("m-device-type",athenaBasicHttpRequestHead.getDeviceType());
        httpHeaders.set("m-device-id",athenaBasicHttpRequestHead.getDeviceId());
        httpHeaders.set("m-source",athenaBasicHttpRequestHead.getSourceFrom());
        httpHeaders.set("digi-middleware-device-id",athenaBasicHttpRequestHead.getIamDeviceId());
        // 获取租户ID
        String tenantId = this.getTenantId(url, athenaBasicHttpRequestHead);
        if (StringUtils.hasLength(tenantId)) {
            httpHeaders.set("routerKey", tenantId);
        }
        // 获取代理proxyToken
        String proxyToken = this.getProxyToken(athenaBasicHttpRequestHead);
        if (StringUtils.hasLength(proxyToken)) {
            httpHeaders.set("digi-proxy-token", proxyToken);
        }
        // 获取多语言配置
        if (StringUtils.hasLength(athenaBasicHttpRequestHead.getAcceptLanguage())) {
            httpHeaders.set("accept-language", athenaBasicHttpRequestHead.getAcceptLanguage());
        } else {
            if (StringUtils.hasLength(athenaBasicHttpRequestHead.getLocale())) {
                httpHeaders.set("accept-language", athenaBasicHttpRequestHead.getLocale().replace("_", "-"));
            }
        }
        try {
            String traceId = TraceIdUtil.get();
            if (org.apache.commons.lang3.StringUtils.isNotBlank(traceId)) {
                httpHeaders.set(TraceIdUtil.TRACE_ID, traceId);
            }
        }
        catch (Exception e) {
            logger.error("getTraceIdEx",e);
        }
        return httpHeaders;
    }

    private HttpEntity<MultiValueMap<String, Object>> processFileHeaders(String url,
                                                                         AthenaBasicHttpRequestHead athenaBasicHttpRequestHead,
                                                                         MultipartFile file, Map<String, ?> requestParam) {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.MULTIPART_FORM_DATA);
        httpHeaders.set("digi-middleware-auth-app", sysEnvConfig.getAppToken());
        httpHeaders.set("digi-middleware-auth-user", athenaBasicHttpRequestHead.getUserToken());
        httpHeaders.set("token", athenaBasicHttpRequestHead.getUserToken());
        httpHeaders.set("client-agent", athenaBasicHttpRequestHead.getClientAgent());
        httpHeaders.set("locale", athenaBasicHttpRequestHead.getLocale());
        // 获取租户ID
        String tenantId = this.getTenantId(url, athenaBasicHttpRequestHead);
        if (StringUtils.hasLength(tenantId)) {
            httpHeaders.set("routerKey", tenantId);
        }
        // 获取代理proxyToken
        String proxyToken = this.getProxyToken(athenaBasicHttpRequestHead);
        if (StringUtils.hasLength(proxyToken)) {
            httpHeaders.set("digi-proxy-token", proxyToken);
        }
//        httpHeaders
        // 构建请求体
        MultiValueMap<String, Object> body = new LinkedMultiValueMap<>();
        body.add("file", file.getResource());
        if (requestParam != null) {
            for (Map.Entry<String, ?> entry : requestParam.entrySet()) {
                body.add(entry.getKey(), entry.getValue());
            }
        }
        return new HttpEntity<>(body, httpHeaders);
    }

    private HttpEntity<MultiValueMap<String, Object>> processFileRequest(String url,
                                                                         AthenaBasicHttpRequestHead athenaBasicHttpRequestHead,
                                                                         List<MultipartFile> files, Map<String, ?> requestParam) {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.MULTIPART_FORM_DATA);
        httpHeaders.set("digi-middleware-auth-app", sysEnvConfig.getAppToken());
        httpHeaders.set("digi-middleware-auth-user", athenaBasicHttpRequestHead.getUserToken());
        httpHeaders.set("token", athenaBasicHttpRequestHead.getUserToken());
        httpHeaders.set("client-agent", athenaBasicHttpRequestHead.getClientAgent());
        httpHeaders.set("locale", athenaBasicHttpRequestHead.getLocale());
        // 获取租户ID
        String tenantId = this.getTenantId(url, athenaBasicHttpRequestHead);
        if (StringUtils.hasLength(tenantId)) {
            httpHeaders.set("routerKey", tenantId);
        }
        // 获取代理proxyToken
        String proxyToken = this.getProxyToken(athenaBasicHttpRequestHead);
        if (StringUtils.hasLength(proxyToken)) {
            httpHeaders.set("digi-proxy-token", proxyToken);
        }
        // 构建请求体
        MultiValueMap<String, Object> body = new LinkedMultiValueMap<>();
        if(files != null){
            for(MultipartFile file:files)
            body.add("files", file.getResource());
        }
        if (requestParam != null) {
            for (Map.Entry<String, ?> entry : requestParam.entrySet()) {
                body.add(entry.getKey(), entry.getValue());
            }
        }
        // 构建请求体
        return new HttpEntity<>(body, httpHeaders);
    }

    /**
     * 智驱中台API调用封装请求头时，获取租户ID
     *
     * @param url
     * @param athenaBasicHttpRequestHead
     * @return
     */
    public String getTenantId(String url, AthenaBasicHttpRequestHead athenaBasicHttpRequestHead) {
        // 从基础头部数据模型 中获取
        String tenantId = athenaBasicHttpRequestHead.getTenantId();
        if (StringUtils.hasLength(tenantId)) {
            return tenantId;
        }
        // 从 TransmittableThreadLocal 缓存中获取
        if (StringUtils.hasLength(tenantId = AppRequestContext.getContextEntity().getTenantId())) {
            return tenantId;
        }
        // 获取动态开关配置是否开启
        if (BooleanUtils.isFalse(jvmCache.getAthenaApiSwitch())) {
            return null;
        }
        // 判断 请求API的URL是否在强校验URL列表中
        if (!AthenaCheckApiEnum.isExist(url)) {
            return null;
        }
        // 存在，抛出异常ServiceException
        throw new ServiceException("API request header tenantId is empty! url：" + url);
    }

    public String getProxyToken(AthenaBasicHttpRequestHead athenaBasicHttpRequestHead) {
        // 从基础头部数据模型 中获取
        String proxyToken = athenaBasicHttpRequestHead.getProxyToken();
        if (StringUtils.hasLength(proxyToken)) {
            return proxyToken;
        }
        // 从 TransmittableThreadLocal 缓存中获取
        if (StringUtils.hasLength(proxyToken = AppRequestContext.getContextEntity().getProxyToken())) {
            return proxyToken;
        }
        return null;
    }

    private HttpHeaders processHeaders(DigiwhaleBasicHttpRequestHead digiwhaleBasicHttpRequestHead) {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.set("Authorization", digiwhaleBasicHttpRequestHead.getAuthorization());
        return httpHeaders;
    }

    @Override
    public <T> ResponseEntity<T> doGet(String url, Class<T> clazz) {
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.GET, HttpEntity.EMPTY, clazz);
        checkResponse(url, responseEntity, "zh_CN");
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doGet(String url, ParameterizedTypeReference<T> responseType) {
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.GET, HttpEntity.EMPTY, responseType);
        checkResponse(url, responseEntity, "zh_CN");
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doGet(String url,
                                       AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, Map<String, ?> uriVariables,
                                       Class<T> clazz) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.GET, requestEntity, clazz, uriVariables);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doGet(String url,
                                       AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, Map<String, ?> uriVariables,
                                       ParameterizedTypeReference<T> responseType) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.GET, requestEntity, responseType,
                        uriVariables);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doGet(String url,
                                       DigiwhaleBasicHttpRequestHead digiwhaleBasicHttpRequestHead, Map<String, ?> uriVariables,
                                       Class<T> clazz) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(digiwhaleBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.GET, requestEntity, clazz,
                        uriVariables);
        checkResponse(url, responseEntity, (String) uriVariables.get("locale"));
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doGet(String url,
                                       AthenaBasicHttpRequestHead athenaBasicHttpRequestHead,
                                       ParameterizedTypeReference<T> responseType) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.GET, requestEntity, responseType);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, Map<String, ?> uriVariables,
                                        Class<T> clazz) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, clazz, uriVariables);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, String bodyPayload, Class<T> clazz) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(bodyPayload, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, clazz);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, String bodyPayload,
                                        ParameterizedTypeReference<T> responseType) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(bodyPayload, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, responseType);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, Map<String, ?> uriVariables,
                                        ParameterizedTypeReference<T> responseType) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, responseType,
                        uriVariables);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead,
                                        ParameterizedTypeReference<T> responseType) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, responseType);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead, Class<T> clazz) {
        HttpEntity<String> requestEntity =
                new HttpEntity<>(null, this.processHeaders(url, athenaBasicHttpRequestHead));
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, clazz);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @SneakyThrows
    public void checkResponse(String url, ResponseEntity responseEntity, String locale) {
        //当为以下链接的时候，不需要校验
        if (url.contains("/oauth/oauth/token")) {
            return;
        }
        if (responseEntity.getStatusCode() == HttpStatus.UNAUTHORIZED) {
            if (!StringUtils.hasLength(locale)) {
                locale = "zh_CN";
            }
            String message = "登录过期";
            if (url.contains("/api/iam/v2/identity/login")) {
                // 2024/11/6,只处理登录接口的返回错误信息值
                try {
                    Map<String, Object> body = JsonUtil.objectToJavaObject(responseEntity.getBody(), new TypeReference<Map<String, Object>>() {
                    });
                    message = (String) body.getOrDefault("errorMessage", "登录过期");
                } catch (Exception ignored) {

                }
            }
            if (url.contains("/identity/token/analyze")) {
                try {
                    Map<String, Object> body = JsonUtil.objectToJavaObject(responseEntity.getBody(), new TypeReference<Map<String, Object>>() {
                    });
                    String errorCode = (String) body.getOrDefault("errorCode", "");
                    //401    21006    您的企业尚未购买此应用
                    //401    21008    您的企业应用授权已到期
                    //token 不过期
                    if (Arrays.asList("21006", "21008").contains(errorCode)) {
                        return;
                    }
                } catch (Exception ignored) {
                }
            }
            throw new Exception(LocaleUtil.getMobileTextByKey(locale, message));
        }
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url, ParameterizedTypeReference<T> responseType) {
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, HttpEntity.EMPTY, responseType);
        checkResponse(url, responseEntity, "zh_CN");
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead,
                                        MultipartFile file,
                                        Map<String, ?> uriVariables, Class<T> clazz) {
        HttpEntity<MultiValueMap<String, Object>> requestEntity =
                this.processFileHeaders(url, athenaBasicHttpRequestHead, file, uriVariables);
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, clazz);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }

    @Override
    public <T> ResponseEntity<T> doPost(String url,
                                        AthenaBasicHttpRequestHead athenaBasicHttpRequestHead,
                                        List<MultipartFile> files, Map<String, ?> requestParam,
                                        ParameterizedTypeReference<T> responseType) {
        HttpEntity<MultiValueMap<String, Object>> requestEntity =
                this.processFileRequest(url, athenaBasicHttpRequestHead, files,requestParam);
        ResponseEntity<T> responseEntity =
                this.restTemplate.exchange(url, HttpMethod.POST, requestEntity, responseType);
        checkResponse(url, responseEntity, athenaBasicHttpRequestHead.getLocale());
        return responseEntity;
    }


}