package com.digiwin.mobile.mobileuibot.common.localization;

import com.digiwin.mobile.mobileuibot.common.context.AppContext;
import com.digiwin.mobile.mobileuibot.common.context.SpringContextHolder;
import com.digiwin.mobile.mobileuibot.locale.service.LocaleService;
import com.digiwin.mobile.mobileuibot.locale.service.impl.LocaleServiceImpl;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import static com.digiwin.mobile.mobileuibot.common.localization.LocalekeyPrefixEnum.*;

/**
 * <p>功能描述：</p>
 * <p>Copyright(c) Digiwin Mobile Technology Co., LTD </p>
 *
 * @FileName: LocaleUtil
 * @Author: Zaregoto
 * @Date: 2021/5/23 21:48
 */
public final class LocaleUtil {

    private static LocaleService localeService = SpringContextHolder.getBean(LocaleServiceImpl.class);

    private static final List<String> PREFIXES = Arrays.asList(DJ_MOBILE_KEY.getValue(), DJ_KEY.getValue());

    /**
     * 通用方法获取文字多语言资源。查找顺序：
     * 1.先查找App端设置的多语言
     * 2.若查不到，则查找PC端设置的多语言
     * 3.还查不到，则返回key本身
     *
     * @param locale
     * @param key
     * @return
     */
    public static String getMobileTextByKey(String locale, String key) {
        if (key == null || key.trim().isEmpty()) {
            return "";
        }
        //兜底-将传进来的key做转换
        if (key.startsWith(I18N_DJ_MOBILE_KEY.getValue())) {
            key = key.substring(I18N_DJ_MOBILE_KEY.getValue().length(), key.length() - 1);
        }
        key = key.replace(DJ_MOBILE_KEY.getValue(), "")
                .replace(DJ_KEY.getValue(), "")
                .replace("mobile-", "");

        Map<String, String> resource = AppContext.getDefaultI18nResourceMap().get(locale);
        for (String prefix : PREFIXES) {
            String i18nKey = prefix + key;
            // 优先检查简体中文资源,key可能是pageID，可能是个英文，可能是某个中文
            Map<String, String> resourceZh = AppContext.getDefaultI18nResourceMap().get(Locale.SIMPLIFIED_CHINESE.toString());
            if (resourceZh != null) {
                String textValue = resourceZh.get(i18nKey);
                if (textValue != null) {
                    // 找到枚举/英文真实的名字，由key转为newValue
                    String newValue = textValue;
                    if (resource != null) {
                        textValue = resource.get(i18nKey);
                        if (textValue != null) {
                            return textValue;
                        }
                    }
                    return localeService.getLanguageValue(locale, newValue);
                }
            }
        }
        return localeService.getLanguageValue(locale, key);
    }

    /**
     * 获取页面中元素的多语言。key为数据库pageSetting配置中，原有的key。<br>
     * 若查不到，则返回配置的key中文字的部分。<br>
     * 举例：数据库中配置的key是<code>${i18n-dj-mobile-打印}</code>，如果查不到对应的多语言结果，则返回配置中的“<code>打印</code>”两个字。
     *
     * @param locale
     * @param databaseKey 数据库中配置的key。比如：${i18n-dj-mobile-打印}
     * @return
     */
    public static String getMobileTextByDatabaseKey(String locale, String databaseKey) {
        return getMobileTextByKey(locale, databaseKey);
    }

    /**
     * 返回移动端展示带单位的文字，仅支持简单的动宾短语。目前针对英语做了单词间空格，以及单复数的简单处理。
     * 使用举例如下：<br/>
     * 逾期3天<br/>
     * prefix=逾期
     * amount=3
     * unit=天
     * isCurrencyUnit=false
     * suffix=""
     * 15天工期<br/>
     * prefix=""
     * amount=15
     * unit=天
     * isCurrencyUnit=false
     * suffix=工期
     * 倒计时3天后完成<br/>
     * prefix=倒计时
     * amount=3
     * unit=天
     * isCurrencyUnit=false
     * suffix=后完成
     * 利润为3000元<br/>
     * prefix=利润为
     * amount=3000
     * unit=元
     * isCurrencyUnit=true
     * suffix=""
     *
     * @param locale         语言别
     * @param prefix         数量前缀文字
     * @param amount         数量文字
     * @param unit           数量单位
     * @param isCurrencyUnit 单位是否是货币。货币单位不做处理
     * @param suffix         数量单位后缀文字
     * @return
     */
    public static String getMobileTextWithUnit(String locale,
                                               String prefix, String amount, String unit, boolean isCurrencyUnit, String suffix) {
        if (locale.equals(Locale.US.toString())) {
            int amountInt = Integer.parseInt(amount);
            String newUnit;
            if (amountInt > 1 && !isCurrencyUnit) {
                newUnit = unit + "s";
            } else {
                newUnit = unit;
            }
            String newPrefix;
            if (!prefix.trim().isEmpty()) {
                newPrefix = prefix + " ";
            } else {
                newPrefix = prefix;
            }
            String newSuffix;
            if (!suffix.trim().isEmpty()) {
                newSuffix = " " + suffix;
            } else {
                newSuffix = suffix;
            }
            return newPrefix + amount + " " + newUnit + newSuffix;
        } else {
            return prefix + amount + unit + suffix;
        }
    }

    /**
     * 获取各种语言下，移动端展示的文字短语
     *
     * @param locale        语言别
     * @param prefix        字段前缀文字
     * @param fieldShowName 字段展示文字
     * @param suffix        字段后缀文字
     * @return
     */
    public static String getPhrasedMobileText(String locale, String prefix, String fieldShowName, String suffix) {
        if (locale.equals(Locale.US.toString())) {
            return prefix + " " + fieldShowName + " " + suffix;
        }
        return prefix + fieldShowName + suffix;
    }
}